# -*- coding: utf-8 -*-

from __future__ import absolute_import

import json
import logging
import re
from datetime import datetime, timedelta
from string import Formatter

import pytz
from django.core.exceptions import ValidationError
from django.core.validators import RegexValidator
from django.db import models
from django.utils import translation
from django.utils.deconstruct import deconstructible
from django.utils.functional import cached_property
from django.utils.http import urlquote
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models_utils.i18n import RouteLTitle, NoneTranslations
from travel.avia.library.python.common.precache.manager import PrecachingManager
from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.utils.date import RunMask, group_days, smart_localize
from travel.avia.library.python.common.utils.fields import CodeCharField, TrimmedCharField, TrimmedTextField, ThreadCalendarField
from travel.avia.library.python.common.xgettext.i18n import gettext


log = logging.getLogger(__name__)


EXPRESS_TYPE_CHOICES = (
    ('', _(u"Обычный рейс")),
    ('express', _(u"Экспресс")),
    ('aeroexpress', _(u"Аэроэкспресс"))
)


@deconstructible
class StringTemplateValidator(object):
    def __init__(self, fields):
        self._fields = set(fields)

    def __call__(self, value):
        try:
            parsed_value = list(Formatter().parse(value))
        except ValueError as e:
            raise ValidationError(_(u"Ошибка в шаблоне {!r}").format(e))

        for _literal, field_name, _format, _conversion in parsed_value:
            if not (field_name is None or field_name in self._fields):
                raise ValidationError(
                    _(u"Неизвестное значение {!r}").format(field_name)
                )

    # Нужно, чтобы джанга не дублировала миграции
    def __eq__(self, other):
        return self._fields == other._fields


class SaleURLFormatter(Formatter):
    def format_field(self, value, format_spec):
        formatted = super(SaleURLFormatter, self).format_field(value, format_spec)

        return urlquote(formatted)


class BaseSupplier(models.Model):
    u""" Поименование рейсов при импорте """
    old_approved_codes = [
        u'airport_donetsk', u'airport_kharkov', u'airport_simferopol', u'arch_av',
        u'avto_siktivcar', u'a_linii', u'b2_msq', u'bb_travel', u'BS_Tushinskaya',
        u'chel_gortrans', u'dyc_ast', u'ekb_northav', u'IP_Bezuglov', u'IP_Stachanov',
        u'IP_Vildanova', u'Kostroma_avia', u'kvc_tula', u'maykop_av',
        u'middle_station_route_af', u'msk_subway', u'next_bus_tour',
        u'Nordexpress_Rybinsk', u'STK_Parus', u'Tarasovskoe_ATP', u'tis_from_hp', u'zd_syktyvkar',
    ]

    title = models.CharField(verbose_name=_(u'название организации поставщика'),
                             max_length=100, blank=False)
    code = CodeCharField(verbose_name=_(u'код организации поставщика'),
                         max_length=100, blank=False, unique=True, null=False,
                         validators=[RegexValidator(
                             ur'(?u)^([0-9a-zA-Z-]*|' + u'|'.join(re.escape(oc) for oc in old_approved_codes) + ur')$',
                             _(u"В коде поставщика могут учавствовать только "
                               u"английские буквы цифры и знак минус"))])

    lmt = models.DateTimeField(verbose_name=_(u'дата последнего импорта'),
                               null=False, blank=False, auto_now_add=True,
                               help_text=_(u'используется только для ТИС'))
    filename = models.CharField(verbose_name=_(u'файл для импорта'),
                                max_length=100, null=True, blank=True)

    hide_in_filters = models.BooleanField(_(u"Не показывать в фильтрах"), default=False)

    exclude_from_external_api = models.BooleanField(_(u"Исключить из внешнего API"), default=False)

    can_buy_ru = models.BooleanField(
        _(u'Разрешить продажу билетов (ru)'), blank=False, null=False, default=False
    )

    can_buy_ua = models.BooleanField(
        _(u'Разрешить продажу билетов (ua)'), blank=False, null=False, default=False
    )

    can_buy_tr = models.BooleanField(
        _(u'Разрешить продажу билетов (tr)'), blank=False, null=False, default=False
    )

    logo = models.ImageField(
        _(u'Логотип'), blank=True, null=True, default=None,
        upload_to='data/supplier_logo'
    )

    sale_url_template = models.CharField(
        _(u'Шаблон ссылки'), max_length=1024, blank=True, null=False, default=u'',
        help_text=u'Например, https://example.org/buy/?from={from_code}&to={to_code}&date={departure:%Y-%m-%d}&time={departure:%H:%M}',
        validators=[
            StringTemplateValidator(fields=('from_code', 'to_code', 'departure'))
        ]
    )

    sale_start_days = models.IntegerField(
        _(u'Дни продажи'), blank=True, null=True,
        help_text=_(u'За сколько суток до отправления начинается продажа')
    )

    sale_stop_hours = models.IntegerField(
        _(u'Часы окончания продажи'), blank=True, null=True,
        help_text=_(u'За сколько часов до отправления заканчивается продажа')
    )

    objects = PrecachingManager(keys=['pk', 'code'])

    __codes_regexp = None

    def __unicode__(self):
        return self.title

    @classmethod
    def codes_regexp(cls):
        if not cls.__codes_regexp:
            cls.__codes_regexp = re.compile("(%s)" % "|".join([s.code for s in cls.objects.all()]))

        return cls.__codes_regexp

    def is_sale_enabled(self, national_version):
        return getattr(self, 'can_buy_{}'.format(national_version), False)

    def build_sale_url(self, *args, **kwargs):
        return SaleURLFormatter().format(self.sale_url_template, *args, **kwargs)

    class Meta:
        verbose_name = _(u'поставщик данных')
        verbose_name_plural = _(u'поставщики данных')
        app_label = 'www'
        abstract = True


class BaseRThread(models.Model):
    """ Нитка - группа вагонов поезда, следующая неразрывно или
        Другая модель самолёта.
    """
    route = models.ForeignKey('Route', verbose_name=_(u'рейс'))
    old_thread_number = models.CharField(
        verbose_name=_(u'Старый номер нитки'), max_length=100, db_column='number',
        null=False, blank=True, default='',
    )
    title = models.CharField(verbose_name=_(u'название'), max_length=100,
                             null=True, blank=False)
    is_manual_title = models.BooleanField(verbose_name=u"Название заполнено вручную", default=True)

    title_tr = TrimmedCharField(_(u'Turkish title'), max_length=100, db_index=True, null=True, blank=True)
    title_tr_author = CodeCharField(max_length=100, null=True)
    title_uk = TrimmedCharField(_(u'Украинское ручное название'), max_length=100, db_index=True, null=True, blank=True)
    title_uk_author = CodeCharField(max_length=100, null=True)

    title_common = models.CharField(verbose_name=_(u'название'), max_length=500,
                                    null=True, blank=False)
    title_short = models.CharField(verbose_name=_(u'название короткое'),
                                   max_length=100, null=True, default=None, blank=True)
    uid = models.CharField(verbose_name=_(u'номер в латинице'), max_length=100, unique=True,
                           null=True, blank=True, help_text=_(u"рассчитывается скриптом"))

    tz_start_time = models.TimeField(verbose_name=_(u'время выезда(вылета) с учетом временной зоны'))
    time_zone = models.CharField(verbose_name=_(u'временная зона'), max_length=30)

    # Поля интервальных ниток
    period_int = models.IntegerField(_(u'Интервал между отправлениями рейсов, в минутах'),
                                     null=True, blank=True, default=None)
    begin_time = models.TimeField(_(u'Начало периода'), null=True, blank=True, default=None)
    end_time = models.TimeField(_(u'Конец периода'), null=True, blank=True, default=None)
    density = TrimmedCharField(_(u'Регулярность хождения в интервале'),
                               max_length=200, null=False, blank=True, default='')

    # Поле для обратных псевдо ниток
    pseudo_data = models.TextField(_(u'Данные для построения комментария для обратных псевдорейсов'),
                                   blank=True, default='')

    comment = TrimmedTextField(_(u'Комментарий'), null=False, blank=True, default='')

    type = models.ForeignKey('RThreadType', verbose_name=_(u'тип'), default=None)
    t_model = models.ForeignKey('TransportModel', verbose_name=_(u'модель'),
                                null=True, blank=True)

    year_days = ThreadCalendarField(verbose_name=_(u'дни хождений'), default=RunMask.EMPTY_YEAR_DAYS,
                                    db_column='tz_year_days')

    has_extrapolatable_mask = models.BooleanField(verbose_name=_(u'экстраполируемые дни хождения'), default=True)
    changed = models.BooleanField(verbose_name=_(u"необходим пересчет табло и поиска"),
                                  null=False, default=False)
    path_and_time_unchanged = models.BooleanField(_(u"Путь и время не менялись"), blank=True, default=False,
                                                  help_text=_(u"можно не пересчитывать нодерот"))
    recount_station_schedule = models.BooleanField(verbose_name=_(u"пересчитать расписание на станциях"),
                                                   null=False, default=False)

    template_text = models.CharField(verbose_name=_(u"Текст шаблонов дней хождений(старое)"),
                                     help_text=_(
                                         u"шаблоны дней хождений через точку с запятой, если есть переключения суток в рейсе, например, \"ежедневно;ежедневно\"."),
                                     max_length=200, default=None,
                                     blank=True, null=True)

    template_start = models.DateField(verbose_name=_(u'Начало хождения шаблона'), null=True,
                                      blank=True)
    template_end = models.DateField(verbose_name=_(u'Конец хождения шаблона'), null=True,
                                    blank=True)
    template_code = models.CharField(verbose_name=_(u"Код шаблона дней хождений"),
                                     max_length=100, default=None,
                                     blank=True, null=True)
    translated_manual_days_texts = models.TextField(_(u'Тексты дней хождения сгенерированные из кода шаблона А.Ф.'),
                                                    blank=True, default='')

    PRECALC_TEXT_DELIMITER = u";"

    translated_days_texts = models.TextField(_(u'Предрасчитанные тексты дней хождения'), blank=True, default='')
    translated_except_texts = models.TextField(_(u'Предрасчитанные тексты исключений дней хождения'), blank=True,
                                               default='')

    basic_thread = models.ForeignKey("RThread", verbose_name=_(u'основная нитка'),
                                     null=True, blank=True, default=None,
                                     related_name='thread_changes')
    import_uid = models.CharField(verbose_name=_(u"уникальный номер нитки генерируемый на основе рейса нитки"),
                                  max_length=255, default=None,
                                  blank=True, null=True, unique=True,
                                  editable=False)

    tariff_type = models.ForeignKey("www.TariffType", null=True, default=None,
                                    blank=True,
                                    verbose_name=_(u'тип цен на электрички'))
    supplier = models.ForeignKey('Supplier', default=None, null=True, blank=True,
                                 verbose_name=_(u"поставщик нитки"))
    is_circular = models.BooleanField(verbose_name=_(u"нитка является кольцевой"),
                                      default=False)
    hidden = models.BooleanField(verbose_name=_(u'нитка скрыта'), default=False)

    is_combined = models.BooleanField(_(u"Нитка согласованного поезда"), default=False)

    t_subtype = models.ForeignKey('www.TransportSubtype', null=True, default=None, blank=True,
                                  verbose_name=_(u'Подтип транспорта'))

    show_in_alldays_pages = models.BooleanField(_(u'Показывать в поиске и расписании на все дни'),
                                                default=True)
    t_type = models.ForeignKey('TransportType', verbose_name=_(u'тип транспорта'))

    company = models.ForeignKey('Company', verbose_name=_(u'компания-перевозчик'),
                                default=None, null=True, blank=True)
    number = models.CharField(verbose_name=_(u'номер'), max_length=100, db_index=True,
                              null=False, blank=True, default=u"", db_column='route_number')
    hidden_number = models.CharField(verbose_name=_(u'скрытый номер'), max_length=100, null=False, blank=True,
                                     default=u"")

    express_type = models.CharField(verbose_name=_(u"тип экспресса"),
                                    choices=EXPRESS_TYPE_CHOICES,
                                    max_length=20, null=True, blank=True,
                                    default=None)
    express_lite = models.ForeignKey('ExpressTypeLite', verbose_name=_(u'подтип экспресса'),
                                     null=True, blank=True, default=None)
    reversed_number = models.CharField(verbose_name=_(u'номер для поиска'), max_length=100,
                                       db_index=True, editable=False, null=False, default=u'')
    schedule_plan = models.ForeignKey('TrainSchedulePlan', verbose_name=_(u"график поездов"),
                                      default=None, blank=True, null=True)
    ordinal_number = models.PositiveIntegerField(_(u'Порядковый номер нитки, используется при генерации uid'),
                                                 null=False, blank=False)
    modified_at = models.DateTimeField(_(u'Дата-время изменения'), auto_now=True, null=False, blank=True)
    L_title = RouteLTitle.L_field()

    def get_thread_tz_start_date(self, start_date, tz=None):
        """
        Нитка по временной зоне tz, стартует в день start_date.
        Возвращаем день старта во временной зоне self.pytz
        """

        if not tz or tz == self.pytz:
            return start_date

        thread_tz_start_dt = smart_localize(datetime.combine(start_date, self.tz_start_time), self.pytz)
        tz_start_dt = thread_tz_start_dt.astimezone(tz)

        tz_start_date = tz_start_dt.date()
        if tz_start_date == start_date:
            return start_date

        if tz_start_date > start_date:
            tz_start_dt -= timedelta(1)
            return tz_start_dt.astimezone(self.pytz).date()

        if tz_start_date < start_date:
            tz_start_dt += timedelta(1)
            return tz_start_dt.astimezone(self.pytz).date()

    def get_run_date_list(self):
        u""" Список дней по которым ходит нитка"""
        return RunMask(self.year_days, today=environment.today()).dates()

    def runs_at(self, dt):
        return RunMask.runs_at(self.year_days, dt)

    def first_run(self, today):
        return RunMask.first_run(self.year_days, today)

    @property
    def cancel(self):
        return self.type.code == 'cancel'

    @property
    def update(self):
        """Нитка-апдейт"""

        return self.type.code == 'change'

    @property
    def assignment(self):
        return self.type.code == 'assignment'

    @property
    def is_interval(self):
        from travel.avia.library.python.common.models.schedule import RThreadType

        return self.type_id == RThreadType.INTERVAL_ID

    def manual_text(self, shift):
        if self.template_text:
            manual_texts = self.template_text.split(self.PRECALC_TEXT_DELIMITER)

            # RASP-10764 ручной шаблон "ежедневно" отображать независимо от смещения
            if u'ежедневно' in manual_texts:
                return u'ежедневно'

            try:
                # Если ручной текст начинается с минуса, значит он включает шаблон на сутки назад
                if manual_texts[0].startswith('-'):
                    manual_texts[0] = manual_texts[0][1:]
                    return manual_texts[shift + 1]
                else:
                    # Иначе считаем, что такого шаблона там нет
                    if shift >= 0:
                        return manual_texts[shift]
            except IndexError:
                pass

        return ''

    def plain_days_text(self, shift, lang='ru'):
        manual_text = self.manual_text(shift)
        if manual_text:
            return manual_text

        shifted_day_texts = None

        if self.translated_manual_days_texts:
            manual_texts = json.loads(self.translated_manual_days_texts)
            if manual_texts:
                try:
                    shifted_day_texts = manual_texts[str(shift)]
                except KeyError:
                    pass

        if not shifted_day_texts:
            texts = json.loads(self.translated_days_texts) if self.translated_days_texts else []
            try:
                shifted_day_texts = texts[shift + 1]
            except IndexError:
                pass

        if shifted_day_texts:
            return shifted_day_texts[lang] if lang in shifted_day_texts else shifted_day_texts['ru']

        return ''

    def except_days_text(self, shift):
        texts = json.loads(self.translated_except_texts) if self.translated_except_texts else []

        try:
            shift_texts = texts[shift + 1]
        except IndexError:
            shift_texts = None

        if shift_texts:
            return shift_texts

        return ''

    def L_days_text(
            self,
            shift,
            except_separator="<br />",
            extra_link=None,
            lang=None,
            html=True,
            template_only=False,
            next_plan=None,
            show_days=False,
            show_all_days=False,
            thread_start_date=None
    ):
        """
        Для получения дней хождения по станции на сегодня,
        вычисляем дату старта нитки так, чтобы рейс прошел
        через эту станцию сегодня по местному времени.

        shift - сдвиг даты между локальной датой отправления/прибытия на станцию и
            датой старта нитки во временной зоне нитки.

        thread_start_date - для формирования текста дней хождения нужно передавать
            дату старта нитки во временной зоне нитки (thread_start_date),
            потому что для одной и той же нитки может быть разный shift
            до и после переключения
        """

        if lang is None:
            lang = translation.get_language()

        return self.days_text(
            shift,
            except_separator=except_separator,
            extra_link=extra_link,
            html=html,
            template_only=template_only,
            next_plan=next_plan,
            lang=lang,
            show_days=show_days,
            show_all_days=show_all_days,
            thread_start_date=thread_start_date
        )

    def days_text(
            self,
            shift,
            except_separator="<br />",
            extra_link=None,
            html=True,
            lang='ru',
            template_only=False,
            next_plan=None,
            show_days=False,
            show_all_days=False,
            thread_start_date=None
    ):
        """ Дни хождения нитки в текстовом виде
            shift - сдвиг в днях от дней отправления
            except_separator - строка, отделяющая основную маску от слов "кроме"
        """

        if thread_start_date is None:
            log.error(u'Для формирования текста дней хождения нужно передавать '
                      u'дату старта нитки во временной зоне нитки, потому что '
                      u'для одной и той же нитки может быть разный shift '
                      u'до и после переключения')

            thread_start_date = environment.now_aware().astimezone(self.pytz).date()

        if self.cancel and not show_days:
            return u'<span class="b-timetable__except">отменён</span>' if html else u'отменён'

        text = self.plain_days_text(shift, lang)
        if not text and template_only:
            # Не возвращаем дни хождения, если нужны только шаблонные
            return None

        update_thread = self.update or self.assignment or self.cancel

        if not text or update_thread:
            # Все даты хождения
            formatted_text = RunMask(
                self.year_days, today=thread_start_date
            ).format_days_text(
                days_limit=(None if update_thread else not show_all_days),
                shift=shift,
                lang=lang
            )
            if formatted_text is None:
                return ''

            text = unicode(formatted_text)

            if self.cancel:
                return text

            if formatted_text.has_more:
                if extra_link:
                    return [formatted_text.raw, extra_link]

                return text

            return gettext(u"только {days}", lang=lang).format(days=text)

        schedule_plan_appendix = u''
        if self.schedule_plan:
            appendix = self.L_schedule_plan_appendix_text(thread_start_date, next_plan)
            if appendix:
                if html:
                    schedule_plan_appendix = u' <span class="b-timetable__gray">%s</span>' % appendix
                else:
                    schedule_plan_appendix = u' %s' % appendix

        except_ = self.except_days_text(shift)
        if except_:
            except_text = group_days(except_, lang)

            except_clause = gettext(u'кроме {days}', lang=lang).format(days=except_text)

            if html:
                except_clause = u'<span class="b-timetable__except">%s</span>' % except_clause

            return text + schedule_plan_appendix + except_separator + except_clause

        return text + schedule_plan_appendix

    def L_schedule_plan_appendix_text(self, thread_start_date, next_plan):
        from travel.avia.library.python.common.models.schedule import TrainSchedulePlan

        if (
            not self.template_start and self.schedule_plan.appendix_type == TrainSchedulePlan.APPENDIX_FROM or
            not self.template_end and self.schedule_plan.appendix_type == TrainSchedulePlan.APPENDIX_TO
        ):
            return self.schedule_plan.get_L_appendix(thread_start_date, next_plan)

        return None

    def L_days_text_dict(
            self,
            shift,
            thread_start_date,
            next_plan=None,
            show_days=False,
            show_all_days=False,
            lang=None
    ):
        """ Дни хождения нитки в словарике
            shift - сдвиг в днях от дней отправления
        """

        lang = lang or translation.get_language()

        if self.cancel and not show_days:
            return {
                'days_text': gettext(u'отменён')
            }

        text = self.plain_days_text(shift, lang)
        update_thread = self.update or self.assignment or self.cancel

        if not text or update_thread:

            # Все даты хождения
            formatted_text = RunMask(
                self.year_days, today=thread_start_date
            ).format_days_text(
                days_limit=(None if update_thread else not show_all_days),
                shift=shift,
                lang=lang
            )
            if formatted_text is None:
                return {}

            text = unicode(formatted_text)

            if self.cancel:
                return {
                    'days_text': text,
                    'cancel': True
                }

            if formatted_text.has_more:
                return {
                    'days_text': text,
                    'has_more_days': True
                }

            return {
                'days_text': gettext(u"только {days}", lang=lang).format(days=text)
            }

        schedule_plan_appendix = u''
        if self.schedule_plan:
            schedule_plan_appendix = (
                self.schedule_plan.get_L_appendix(thread_start_date, next_plan) or u''
            )
        if schedule_plan_appendix:
            text += u' ' + schedule_plan_appendix

        except_ = self.except_days_text(shift)
        if except_:
            except_days = group_days(except_, lang)

            return {
                'days_text': text,
                'except_days_text': except_days
            }

        return {
            'days_text': text
        }

    @property
    def path(self):
        """Упорядоченный список rtstations"""

        return self.rtstation_set.order_by('id')

    @cached_property
    def pytz(self):
        return pytz.timezone(self.time_zone)

    @property
    def display_path(self):
        """Упорядоченный список rtstations со станциями и без технических остановок"""
        # select_related('station') не вредно, так-как join все-равно осуществляется из-за
        # station__<attr>
        return self.path.filter(station__majority__id__lt=5, station__hidden=False,
                                is_technical_stop=False).select_related('station')

    @property
    def L_title_special(self):
        if self.deluxe_train:
            return self.deluxe_train.L_title

        return NoneTranslations()

    def L_title_special_short(self):
        if self.deluxe_train:
            return self.deluxe_train.L_title_short()

        return u''

    def L_special_transport(self):
        """
        Возвращает Арэроэкспресс или Экспресс
        для соответсвуюших типов транспорта, или None
        """

        if self.is_aeroexpress:
            return gettext(u'аэроэкспресс')

        if self.is_express:
            return gettext(u'экспресс')

    def L_transport_title(self, lang=None):
        return self.L_special_transport() or self.t_type.L_title(lang=lang)

    @property
    def subtype_code(self):
        """
        Код подтипа транспорта
        train - поезд дальнего следования
        suburban - электричка
        aeroex - аэроэкспресс
        bus
        plane
        river
        sea
        """

        if self.is_aeroexpress:
            return 'aeroex'

        return self.t_type.code

    @property
    def is_deluxe(self):
        if self.deluxe_train:
            return self.deluxe_train.deluxe

        return False

    @cached_property
    def deluxe_train(self):
        from travel.avia.library.python.common.models.schedule import DeLuxeTrain
        from travel.avia.library.python.common.models.transport import TransportType

        if self.t_type_id == TransportType.TRAIN_ID:
            return DeLuxeTrain.get_by_number(self.number)

    @property
    def is_express(self):
        return self.express_type == 'express'

    @property
    def is_aeroexpress(self):
        return self.express_type == 'aeroexpress'

    def get_popular_title(self, lang=None):
        return self.L_title(popular=True, lang=lang)

    def get_tariff_type_id(self):
        # TODO заполнять для всех ниток tariff_type_id на этапе импорта
        from travel.avia.library.python.common.models.tariffs import TariffType

        if self.tariff_type_id:
            return self.tariff_type_id

        if self.is_express or self.is_aeroexpress:
            return TariffType.EXPRESS_ID

        return TariffType.DEFAULT_ID

    class Meta:
        abstract = True
        verbose_name = _(u'нитка')
        verbose_name_plural = _(u'нитки')
        unique_together = ('route', 'ordinal_number',)
        app_label = 'www'
        db_table = 'www_rthread'


class BaseRoute(models.Model):
    """ Рейс
    Фактически - это номер, и дополнительная общая информация для всех ниток.
    """

    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u'тип транспорта'))
    comment = models.TextField(verbose_name=_(u'комментарий'),
                               null=True, blank=True, default=u"")
    supplier = models.ForeignKey('www.Supplier', verbose_name=_(u'поставщик'),
                                 null=False, blank=True,
                                 limit_choices_to={'hide_in_filters': False})
    # В базе данных содавать как `script_protected` bool NOT NULL DEFAULT FALSE
    script_protected = models.BooleanField(verbose_name=_(u"не менять при импорте"),
                                           null=False, default=True)
    # TODO: заменить просто на uid в базе
    route_uid = models.CharField(verbose_name=_(u"идентификатор"),
                                 help_text=_(u"идентификатор для однозначного "
                                             u"определения рейса из импорта"),
                                 max_length=100, null=False)
    hidden = models.BooleanField(verbose_name=_(u"рейс скрыт"), null=False,
                                 default=False)

    style = models.TextField(verbose_name=_(u'стиль линии метро'),
                             blank=True, default=None, null=True)
    modified_at = models.DateTimeField(_(u'Дата-время изменения'), auto_now=True, null=False, blank=True)

    class Meta:
        verbose_name = _(u'рейс')
        verbose_name_plural = _(u'рейсы')
        app_label = 'www'
        db_table = 'www_route'
        unique_together = (('route_uid', 't_type', 'supplier'),)
        abstract = True
