# -*- coding: utf-8 -*-

from __future__ import absolute_import

import json
import logging
from datetime import datetime, timedelta

from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.utils.date import RunMask, get_pytz
from travel.avia.library.python.common.utils.fields import TrimmedCharField, ThreadCalendarField

log = logging.getLogger(__name__)


class BaseThreadTariff(models.Model):
    """ Тариф станция-станция """
    thread_uid = models.CharField(verbose_name=_(u"Идентификатор"),
                                  help_text=_(u"Идентификатор для однозначного определения нитки"),
                                  max_length=100, null=False, db_index=True)
    year_days = ThreadCalendarField(verbose_name=_(u'Шаблон дней хождения для дней когда действует эта цена'),
                                    default=u'')
    station_from = models.ForeignKey('www.Station', verbose_name=_(u'Первая станция'),
                                     related_name='route_tariffs_from')
    station_to = models.ForeignKey('www.Station', verbose_name=_(u'Последняя станция'),
                                   related_name='route_tariffs_to')
    tariff = models.FloatField(_(u'Цена'))
    currency = models.CharField(_(u'Валюта'), max_length=10, null=True, default=None, blank=True)
    is_min_tariff = models.BooleanField(_(u'Это минимальная цена поставщика, могут быть и выше'), default=False)
    url = TrimmedCharField(verbose_name=_(u'order Url'),
                           max_length=255, null=False, default='', blank=True)
    data = models.TextField(verbose_name=_(u"Дополнительные данные"),
                            null=False, blank=False, default='')
    settlement_from = models.ForeignKey('www.Settlement', verbose_name=_(u'Город отправлени'),
                                        null=True, related_name='tariffs_from')
    settlement_to = models.ForeignKey('www.Settlement', verbose_name=_(u'Город прибытия'),
                                      null=True, related_name='tariffs_to')
    year_days_from = ThreadCalendarField(verbose_name=_(u'Календарь во временной зоне отправления'),
                                         default=RunMask.EMPTY_YEAR_DAYS)
    time_zone_from = models.CharField(verbose_name=_(u'Временная зона отправления'), max_length=30)
    time_from = models.TimeField(verbose_name=_(u'Время отправления с учетом временной зоны'))
    duration = models.IntegerField(verbose_name=_(u'Время в пути (в минутах).'), null=False, blank=False)
    time_zone_to = models.CharField(verbose_name=_(u'Временная зона прибытия'), max_length=30)
    time_to = models.TimeField(verbose_name=_(u'Время прибытия с учетом временной зоны'))
    supplier = models.ForeignKey('www.Supplier', null=False, blank=False, verbose_name=_(u'Поставщик тарифа'))
    number = models.CharField(verbose_name=_(u'Номер маршрута'), max_length=100, null=False, blank=True, default=u'')
    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u'Тип транспорта'), null=False, blank=False)

    def get_order_data(self):
        return json.loads(self.data) if self.data else {}

    def get_local_departure(self, local_date):
        """
        Возвращает местное время отправления, соответствующее дате local_date
        """
        departure = datetime.combine(local_date, self.time_from)
        local_tz = self.station_from.pytz
        departure_tz = get_pytz(self.time_zone_from)

        if local_tz == departure_tz:
            departure_naive = departure
            departure_local = local_tz.localize(departure_naive)
        else:
            for offset in (0, -1, 1):
                departure_naive = departure + timedelta(days=offset)
                departure_local = departure_tz.localize(departure_naive).astimezone(local_tz)
                if departure_local.date() == local_date:
                    break
            else:
                # невероятный случай, когда два часовых пояса отличаются больше чем на сутки
                return None

        if RunMask.runs_at(self.year_days_from, departure_naive):
            return departure_local

    def __unicode__(self):
        return u'-'.join([self.thread_uid, self.station_from.title,
                          self.station_to.title, unicode(self.tariff)])

    class Meta:
        abstract = True
        verbose_name = _(u'тариф нитка-точка-точка')
        verbose_name_plural = _(u'тарифы нитка-точка-точка')
        app_label = 'www'
        index_together = [
            ['settlement_from', 'settlement_to'],
            ['settlement_from', 'station_to'],
            ['station_from', 'settlement_to'],
            ['station_from', 'station_to'],
        ]
