# -*- coding: utf-8 -*-

"""
Эти хелперы используются в абстрактных моделях и поэтому не должны зависеть от обычных моделей
"""

import logging
import operator
import random
from itertools import islice, groupby, izip

from django.db import models
from django.utils.datastructures import MultiValueDict
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.utils.mysql_try_hard import mysql_try_hard
from travel.avia.library.python.common.xgettext.i18n import dynamic_gettext


log = logging.getLogger(__name__)

TABLO_MANUAL_STATUS_CHOISES = (
    ('use_real_time', _(u"Вичислить по реальному времени")),
    ('cancelled', _(u"Отменен")),
    ('hidden', _(u"Скрыт"))
)
EVENT_CHOISES = (
    ('arrival', _(u"Прибытие")),
    ('departure', _(u"Отправление"))
)


def take(n, iterable):
    "Return first n items of the iterable as a list"
    return list(islice(iterable, n))


def model_iter(qs, chunksize=100, in_chunks=False, shuffle=False, plain_qs=None):
    pks = list(qs.values_list('pk', flat=True))

    if shuffle:
        random.shuffle(pks)

    plain_qs = plain_qs if plain_qs is not None else qs.all().order_by()

    if chunksize == 1:
        for pk in pks:
            yield plain_qs.get(pk=pk)

        return

    pks_iter = iter(pks)

    while 1:
        pks = take(chunksize, pks_iter)

        if not pks:
            break

        objects = mysql_try_hard(plain_qs.in_bulk)(pks)

        if in_chunks:
            yield [objects[pk] for pk in pks if pk in objects]
        else:
            for pk in pks:
                try:
                    yield objects[pk]
                except KeyError:
                    raise


def split_to_bins(records, groupby_key, chunksize=100000, binsize=50000):
    record_iter = model_iter(records, chunksize=chunksize)

    # Нельзя обрабатывать записей меньше, чем все записи треда, но не хочется при этом
    # обрабатывать меньше BINSIZE, ради производительности, поэтому складываем группы в корзину
    # и обрабатываем, когда накопится достаточно

    bin = []

    for grouper, records in groupby(record_iter, key=groupby_key):
        bin.extend(records)

        if len(bin) < binsize:
            continue

        yield bin

        bin = []

    # Обрабатываем последние записи
    if bin:
        yield bin


def _iter_fetch_related(objects, field, model=None, skip_errors=False):
    if model is None:
        model = getattr(objects, 'model', None)

        if model is None:
            raise ValueError(
                'Please pass queryset or specify model directly')

    model_field = getattr(model, field).field
    related_model = model_field.rel.to

    getter = operator.attrgetter(model_field.attname)

    ids = map(getter, objects)
    uniq_ids = list(set(ids) - {None})

    related_objects = related_model.objects.order_by().in_bulk(uniq_ids)

    for obj, related_id in izip(objects, ids):
        try:
            setattr(
                obj, field,
                None if related_id is None else related_objects[related_id]
            )
        except Exception:
            log.exception(
                u'Ошибка установки атрибута %s %s %s %s',
                obj.__class__, obj.pk, field,
                related_objects.get(related_id)
            )

            if not skip_errors:
                raise
        else:
            yield obj


def fetch_related(objects, *fields, **kwargs):
    """ Устанавливает related objects через отдельные запросы """

    for field in fields:
        for _obj in _iter_fetch_related(objects, field,
                                        skip_errors=False, **kwargs):
            pass


def fetch_related_safe(objects, *fields, **kwargs):
    """
        Устанавливает related objects через отдельные запросы, пропуская
        объекты, для которых это не получилось сделать
    """

    for field in fields:
        objects = list(_iter_fetch_related(objects, field,
                                           skip_errors=True, **kwargs))

    return objects


def fetch_children(objects, qs, field):
    pk_getter = operator.attrgetter(getattr(qs.model, field).field.attname)

    d = MultiValueDict()

    for child in qs.filter(**{'%s__in' % field: objects}):
        d.appendlist(pk_getter(child), child)

    return d


class HiddenManagerWrapper(models.Manager):
    """
    Взывает к менеджерам, не возвращая скрытые объекты
    """

    def __init__(self, manager_name):
        super(HiddenManagerWrapper, self).__init__()
        self.original_manager_name = manager_name

    @property
    def original_manager(self):
        return getattr(self.model, self.original_manager_name)

    def get(self, *args, **kwargs):
        if getattr(self.original_manager, 'precached', False):
            obj = self.original_manager.get(*args, **kwargs)
            if obj.hidden:
                raise self.model.DoesNotExist
            else:
                return obj

        return self.get_queryset().get(*args, **kwargs)

    def get_list(self, *args, **kwargs):
        return [o for o in self.original_manager.filter(*args, **kwargs) if not o.hidden]

    def get_queryset(self):
        return self.original_manager.get_queryset().filter(hidden=False)


class DisplayTCodeMixin(object):
    @property
    def display_t_code(self):
        if getattr(self, 'thread', None):
            if self.thread.is_express:
                return 'express'

            if self.thread.is_aeroexpress:
                return 'aeroexpress'

        if getattr(self, 't_model', None):
            return self.t_model.t_type.code

        if getattr(self, 't_type', None):
            return self.t_type.code

        return self.thread.t_type.code

    @classmethod
    def t_code_name(self, code):
        from travel.avia.library.python.common.models.transport import TRANSPORT_TYPES

        source_name = TRANSPORT_TYPES.get(code)

        if source_name:
            return dynamic_gettext(source_name)

        return None


class ConstField(object):
    """
    Дескриптор отдающий константное значение. Можно перекрывать django поле,
    а также использовать как атрибут-константу
    """

    def __init__(self, value):
        self._const_value = value

    def __get__(self, instance, owner):
        return self._const_value

    def __set__(self, instance, value):
        """
        Сеттер необходим для django. В момент получения объекта из базы
        django заполняет значением.
        """
