# -*- coding: utf-8 -*-

"""
Эти хелперы используются в абстрактных моделях и поэтому не должны зависеть от обычных моделей
"""

import logging
import re
import warnings

from django.conf import settings
from django.core.exceptions import ObjectDoesNotExist
from django.db import models
from django.utils import translation
from django.utils.encoding import smart_str
from django.utils.functional import cached_property

from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.models_utils.i18n import L_field
from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.utils.caching import cached
from travel.avia.library.python.common.utils.date import get_pytz, MSK_TZ, get_msk_time_zone, get_timezone_from_point
from travel.avia.library.python.common.utils.geobase import geobase
from travel.avia.library.python.common.utils.text import title_with_preposition
from travel.avia.library.python.common.utils.warnings import RaspDeprecationWarning

log = logging.getLogger(__name__)


RE_MDOT = re.compile(ur'^м.\s*', re.U)


TR_SUGGEST_FULL_TITLE_OVERRIDES = {
    'c31716': [u"Gazimağusa", u"Kuzey Kıbrıs"],
    's9838544': [u"Gazimağusa liman", u"Gazimağusa", u"Kuzey Kıbrıs"],
    's9838502': [u"Girne liman", u"Girne Merkez", u"Kuzey Kıbrıs"],
    'c40775': [u"Lefkoşa", u"Kuzey Kıbrıs"],
    'c41719': [u"Lefkoşa", u"Kuzey Kıbrıs"],
    's9631390': [u"Ercan havalimanı", u"Lefkoşa", u"Kuzey Kıbrıs"],
}

NOTHERN_CYPRUS_CITIES = [
    31716  # Фамагуста
]


class TimeZoneMixin(object):
    # Переводит время без временной зоны из Мск в местное

    @cached_property
    def pytz(self):
        return get_pytz(self.get_tz_name())

    def utcoffset(self):
        return int(environment.now_aware().astimezone(self.pytz).utcoffset().total_seconds())

    def localize(self, msk=None, loc=None):
        if msk:
            if msk.tzinfo is None:
                msk = MSK_TZ.localize(msk)

                if self.pytz == MSK_TZ:
                    return msk

            return msk.astimezone(self.pytz)
        elif loc:
            if loc.tzinfo is None:
                return self.pytz.localize(loc)

            return loc.astimezone(self.pytz)

    def get_msk_datetime(self, loc):
        local_datetime = self.localize(loc=loc)
        return local_datetime.astimezone(MSK_TZ)

    def get_local_datetime(self, msk):
        return self.localize(msk=msk)

    def local_time(self, localized_dt):
        # FIXME: навание не соответствует типу
        return localized_dt.astimezone(self.pytz)

    def get_time_zone(self, **kwargs):
        """Возвращает временную зону с учетом контекстного времени

        Keyword Arguments:
        msk_datetime: Московское время, тип datetime
        local_datetime: местное время, тип datetime
        Одно из времен указать обязательно
        """
        return get_msk_time_zone(self, **kwargs)

    def get_time_zone_offset_from_msk_in_minutes(self, **kwargs):
        """Возвращает временную зону с учетом контекстного времени

        Keyword Arguments:
        msk_datetime: Московское время, тип datetime
        local_datetime: местное время, тип datetime
        Одно из времен указать обязательно
        """
        return get_msk_time_zone(self, **kwargs)

    def get_tz_name(self):
        return get_timezone_from_point(self)


class TransLocalMixin(object):
    def _translocal_country(self, national_version):
        from travel.avia.library.python.common.models.geo import Country
        if not geobase:
            return

        geo_id = self.get_settlement_geo_id() or self.get_region_geo_id()

        if not geo_id:
            log.info(u'У спорного региона %s %s не нашли geo_id города или области',
                     self.id, self.__class__.__name__)
            return

        try:
            country_geo_id = geobase.find_country_id(geo_id, smart_str(national_version))

        except Exception:
            log.exception(u'Не удалось получить транслокальную страну для'
                          u' спорного региона %s национальная версия %s',
                          geo_id, national_version)
            return

        # Геобаза почему-то иногда возращает 0
        if country_geo_id == 0:
            log.error(u'Геобаза вернула geo_id == 0 для страны'
                      u' спорного региона %s национальная версия %s',
                      geo_id, national_version)
            return

        try:
            return Country.objects.get(_geo_id=country_geo_id)
        except Country.DoesNotExist:
            log.exception(u'Не нашли страну по geoid %s'
                          u' спорного города %s национальная версия %s',
                          country_geo_id, geo_id, national_version)

    def translocal_country(self, national_version=None):
        if not self.disputed_territory:
            return self.country

        if not national_version:
            return None

        return self._translocal_country(national_version)


class Geobase_L_title(L_field):
    def contribute_to_class(self, cls, name):
        assert name == 'L_title'

        return super(Geobase_L_title, self).contribute_to_class(cls, name)

    def exact_linguistics(self, obj, lang, case):
        # Значения в базе имеют приоритет над геобазой
        super_form = super(Geobase_L_title, self).exact_linguistics(obj, lang, case)

        if super_form:
            return super_form

        if obj._geo_id:
            linguistics = self.get_geobase_linguistics(obj._geo_id, lang)
            if linguistics:
                try:
                    return getattr(linguistics, case)

                except AttributeError:
                    pass

    def get_geobase_linguistics(self, geo_id, lang):
        if geobase:
            try:
                return geobase.linguistics(geo_id, str(lang))

            except RuntimeError:
                log.warning(u'Не удалось получить локальные наименования для %s, язык %s', geo_id, lang)


class Point(object):
    @property
    def is_station(self):
        from travel.avia.library.python.common.models_abstract.geo import BaseStation
        return isinstance(self, BaseStation)

    @classmethod
    def model_key_prefix(cls, model):
        from travel.avia.library.python.common.models_abstract.geo import BaseStation
        from travel.avia.library.python.common.models.geo import Settlement, Region, Country

        if issubclass(model, Settlement):
            return 'c'

        elif issubclass(model, BaseStation):
            return 's'

        elif issubclass(model, Region):
            return 'r'

        elif issubclass(model, Country):
            return 'l'

        raise NotImplementedError("Keys for type %s are not supported" % model.__name__)

    @classmethod
    def get_point_key(cls, model, pk):
        prefix = cls.model_key_prefix(model)

        return "%s%s" % (prefix, pk)

    @property
    def point_key(self):
        return self.get_point_key(self.__class__, self.pk)

    @classmethod
    def parse_key(cls, key):
        from travel.avia.library.python.common.models.geo import Station, Settlement, Region, Country

        try:
            prefix, pk = key[0], key[1:]

        except (IndexError, TypeError):
            raise ValueError("Unknown key %r" % key)

        if prefix == 'c':
            return Settlement, pk
        elif prefix == 's':
            return Station, pk
        elif prefix == 'r':
            return Region, pk
        elif prefix == 'l':
            return Country, pk

        raise ValueError("Unknown type prefix %s" % prefix)

    @classmethod
    def get_by_key(cls, key):
        from travel.avia.library.python.common.models.geo import Settlement, Country

        try:
            prefix, geo_id = key[0], key[1:]
        except (IndexError, TypeError):
            raise ValueError("Unknown key %r" % key)

        if prefix == 'g':
            try:
                return Settlement.objects.get(_geo_id=geo_id)
            except Settlement.DoesNotExist:
                pass

            try:
                return Country.objects.get(_geo_id=geo_id)
            except Country.DoesNotExist:
                pass

            raise ObjectDoesNotExist(u'Geo Object with geo_id == %s does not exist' % geo_id)

        model, pk = cls.parse_key(key)

        if hasattr(model, 'hidden_manager'):
            return model.hidden_manager.get(pk=pk)

        return model.objects.get(pk=pk)

    @classmethod
    def get_any_by_key(cls, key):
        model, pk = cls.parse_key(key)

        return model.objects.get(pk=pk)

    @classmethod
    def in_bulk(cls, keys):
        foo = {}

        for key in keys:
            model, pk = cls.parse_key(key)

            foo.setdefault(model, []).append(pk)

        objects = {}

        for model, pks in foo.items():
            prefix = cls.model_key_prefix(model)

            for obj in model.objects.in_bulk(pks).values():
                objects['%s%s' % (prefix, obj.pk)] = obj

        return objects

    def L_title_phrase_from(self, lang=None):
        """
        Получает локализованную фразу 'из города': 'Из Киева', 'from Kiev', 'з Києва'.
        Если не найдется название города в подходящем падеже, то вернется None.
        """
        lang = lang or translation.get_language()

        if lang == 'tr':
            return None

        if lang == 'en':
            title = self.L_title(lang=lang)
            return title_with_preposition(u'from', title)

        if lang == 'ru':
            return self.title_ru_phrase_from

        if lang == 'uk':
            title = self.L_title(case='genitive', fallback=False, lang=lang)
            return title_with_preposition(u'з', title)

    def L_title_phrase_to(self, lang=None):
        """
        Получает локализованную фразу 'в город': 'В Киев', 'to Kiev', 'до Києва'.
        Если не найдется название города в подходящем падеже, то вернется None.
        """
        lang = lang or translation.get_language()

        if lang == 'tr':
            return None

        if lang == 'en':
            title = self.L_title(lang=lang)
            return title_with_preposition(u'to', title)

        if lang == 'ru':
            return self.title_ru_phrase_to

        if lang == 'uk':
            title = self.L_title(case='genitive', fallback=False, lang=lang)
            return title_with_preposition(u'до', title)

        return None

    @property
    def title_ru_phrase_from(self):
        title = self.L_title(lang='ru', fallback=False, case='genitive')
        return title_with_preposition(u'из', title)

    @property
    def title_ru_phrase_in(self):
        title = self.L_title(lang='ru', case='locative')
        return title_with_preposition(self.title_ru_preposition_v_vo_na or u"в", title)

    @property
    def title_ru_phrase_to(self):
        title = self.L_title(lang='ru', fallback=False, case='accusative')
        return title_with_preposition(self.title_ru_preposition_v_vo_na or u"в", title)

    @property
    def title_uk_phrase_in(self):
        return title_with_preposition(
            self.L_title(case='preposition', fallback=False) or u"у",
            self.L_title(case='prepositional')
        )

    @property
    @cached(lambda self: 'find_iata_point_%s%s' % (self.__class__.__name__, self.id),
            timeout=settings.CACHES['default']['LONG_TIMEOUT'])
    def iata_point(self):
        from travel.avia.library.python.common.models.geo import Station, Settlement

        if isinstance(self, Station):
            station = self
            settlement = self.settlement
            if settlement and settlement.iata:
                return settlement
            if station.iata:
                return station

        elif isinstance(self, Settlement):
            stations = self.station_set.filter(t_type__code='plane', hidden=False)
            if stations:
                # У города есть свои аэропорты, берем его код если указан
                settlement = self
                if settlement and settlement.iata:
                    return settlement
                for station in stations:
                    if station.iata:
                        return station
            else:
                # Если до города летают, то должен быть хотя бы один аэропорт,
                # привязанный на прямую или через множественную привязку.
                # Потому игнорируем наличие у города код iata
                # Если он не привязан ни к одному аэропорту
                s2s_set = self.station2settlement_set \
                    .filter(station__t_type__code='plane', station__hidden=False) \
                    .select_related('station')
                for s2s in s2s_set:
                    station = s2s.station
                    settlement = station.settlement
                    if settlement and settlement.iata:
                        return settlement
                    if station.iata:
                        return station

        return None

    @cached(lambda self: 'get_all_iata_airports_%s%s' % (self.__class__.__name__, self.id),
            timeout=settings.CACHES['default']['LONG_TIMEOUT'])
    def get_all_iata_airports(self):
        from travel.avia.library.python.common.models.geo import Station, Settlement

        airports = []

        if isinstance(self, Station):
            settlement = self.settlement
            if not settlement:
                if self.iata and self.t_type.id == TransportType.PLANE_ID:
                    airports.append(self)

                return airports

        elif isinstance(self, Settlement):
            settlement = self
        else:
            return airports

        for airport in settlement.station_set.filter(hidden=False):
            if airport.iata:
                airports.append(airport)

        if not airports:
            for s2s in settlement.station2settlement_set \
                    .filter(station__t_type__code='plane', station__hidden=False) \
                    .select_related('station'):

                if s2s.station.iata:
                    airports.append(s2s.station)

        return airports

    @property
    @cached(lambda self: 'find_sirena_point_%s%s' % (self.__class__.__name__, self.id),
            timeout=settings.CACHES['default']['LONG_TIMEOUT'])
    def sirena_point(self):
        from travel.avia.library.python.common.models.geo import Station, Settlement

        """ Ищет точку с не пустым sirena_id, например, аэропорт города, если у
        города sirena_id не указан. Точка может быть привязана, как напрямую, так и
        через множественную связь.
        Если это аэропорт, то постараемся взять город для этого аэропорта
        """

        if isinstance(self, Station):
            station = self
            settlement = self.settlement
            if settlement and settlement.sirena_id:
                return settlement
            if station.sirena_id:
                return station

        elif isinstance(self, Settlement):
            stations = self.station_set.filter(t_type__code='plane', hidden=False)
            if stations:
                # У города есть свои аэропорты, берем его код если указан
                settlement = self
                if settlement and settlement.sirena_id:
                    return settlement
                for station in stations:
                    if station.sirena_id:
                        return station
            else:
                # Если до города летают, то должен быть хотя бы один аэропорт,
                # привязанный на прямую или через множественную привязку.
                # Потому игнорируем наличие у города sirena_id
                # Если он не привязан ни к одному аэропорту
                s2s_set = self.station2settlement_set \
                    .filter(station__t_type__code='plane', station__hidden=False) \
                    .select_related('station')
                for s2s in s2s_set:
                    station = s2s.station
                    settlement = station.settlement
                    if settlement and settlement.sirena_id:
                        return settlement
                    if station.sirena_id:
                        return station

        return None

    @cached(lambda self: 'point_buscomua_code_%s%s' % (self.__class__.__name__, self.id),
            timeout=settings.CACHES['default']['LONG_TIMEOUT'])
    def get_buscomua_city_code(self):
        from travel.avia.library.python.common.models.geo import Station, Settlement

        """ Ищет точку с не пустым buscomua """
        from travel.avia.library.python.common.importinfo.models.bus import BuscomuaStationCode

        if isinstance(self, Station):
            try:
                return BuscomuaStationCode.objects.filter(station=self).only('city_code')[0].city_code
            except IndexError:
                pass

        elif isinstance(self, Settlement):
            try:
                return BuscomuaStationCode.objects.filter(station__settlement=self).only('city_code')[0].city_code
            except IndexError:
                pass
        else:
            return None

    @cached(lambda self: 'get_swdfactory_stations%s%s' % (self.__class__.__name__, self.id),
            timeout=settings.CACHES['default']['LONG_TIMEOUT'])
    def get_swdfactory_codes(self):
        from travel.avia.library.python.common.models.geo import Station, Settlement, CodeSystem
        """
        Ищет все точки города с не пустым swdfactory
        """

        codesystem = CodeSystem.objects.get(code='swdfactory')

        if isinstance(self, Station):
            code = self.get_code(codesystem)
            if code:
                return [code]
            else:
                return []

        elif not isinstance(self, Settlement):
            return []

        settlement = self

        result = []
        for station in settlement.station_set.filter(hidden=False,
                                                     code_set__code__isnull=False,
                                                     code_set__system=codesystem).only('id'):
            code = station.get_code(codesystem)
            if code:
                result.append(code)

        return result

    @cached(lambda self: 'get_ukrmintrans_stations%s%s' % (self.__class__.__name__, self.id),
            timeout=settings.CACHES['default']['LONG_TIMEOUT'])
    def get_ukrmintrans_codes(self):
        """
        Ищет все точки города с не пустым ukrmintrans
        и важностью больше или равной "Попадает в табло"
        """

        from travel.avia.library.python.common.models.geo import Station, Settlement, CodeSystem

        codesystem = CodeSystem.objects.get(code='ukrmintrans')

        if isinstance(self, Station):
            code = self.get_code(codesystem)
            if code:
                return [code]
            else:
                return []

        elif isinstance(self, Settlement):
            settlement = self
        else:
            return []

        result = []
        for station in settlement.station_set.filter(hidden=False,
                                                     code_set__code__isnull=False,
                                                     code_set__system=codesystem,
                                                     majority__id__lte=2).only('id'):
            code = station.get_code(codesystem)
            if code:
                result.append(code)

        return result

    def get_ukrmintrans_express_codes(self):
        stations = self.get_express_stations()

        return [st.express_id for st in stations]

    def get_railway_tz_points(self):
        railway_tz_points = dict()

        for station in self.get_express_stations():
            railway_tz_points[station.express_id] = station.get_railway_tz_point()

        return railway_tz_points

    @cached(lambda self: 'point_chartex_code_%s%s' % (self.__class__.__name__, self.id),
            timeout=settings.CACHES['default']['LONG_TIMEOUT'])
    def get_chartex_city_code(self):
        """ Ищет chartex_ID """
        from travel.avia.library.python.common.models.geo import Station, Settlement, SettlementCode

        settlement = None

        if isinstance(self, Station):
            settlement = self.settlement

        elif isinstance(self, Settlement):
            settlement = self

        if settlement:
            try:
                return settlement.code_set.get(system__code='chartex').code

            except SettlementCode.DoesNotExist:
                pass

    @cached(lambda self: 'get_express_stations%s%s' % (self.__class__.__name__, self.id),
            timeout=settings.CACHES['default']['LONG_TIMEOUT'])
    def get_express_stations(self):
        """
        Для станции возвращает список из нее, если у нее есть экспресс код, иначе пустой список.
        Для города возвращает список из фейковых станций express,
        если нет фейковых express станций, то возвращает список из главных станций в городе с экспресс кодами,
        если нет главных станций, то возвращает список всех станций с экспресс кодами.
        """
        from travel.avia.library.python.common.models.geo import Station, Settlement

        if isinstance(self, Station):
            if self.express_id:
                return [self]
            else:
                return []

        elif isinstance(self, Settlement):
            settlement = self

        else:
            return []

        stations = settlement.station_set.filter(hidden=False, express_id__isnull=False) \
            .only('express_id', 'majority__code', 'country')

        express_fake_stations = [st for st in stations if st.majority.code == 'express_fake']

        if express_fake_stations:
            return express_fake_stations

        main_in_city_express_stations = [st for st in stations if st.majority.code == 'main_in_city']

        if main_in_city_express_stations:
            return main_in_city_express_stations

        return stations

    def is_in_region_id(self, region_id):
        from travel.avia.library.python.common.models.geo import Station, Settlement

        if isinstance(self, Settlement) or isinstance(self, Station):
            return self.region_id == region_id
        else:
            return False

    def is_in_settlement_id(self, settlement_id):
        from travel.avia.library.python.common.models.geo import Station

        if isinstance(self, Station):
            return self.settlement_id == settlement_id
        else:
            return False

    def get_settlement_geo_id(self):
        from travel.avia.library.python.common.models.geo import Station, Settlement

        if isinstance(self, Station):
            return self.settlement_id and self.settlement._geo_id or None
        elif isinstance(self, Settlement):
            return self._geo_id
        return None

    def get_region_geo_id(self):
        from travel.avia.library.python.common.models.geo import Station, Settlement, Region

        if isinstance(self, Station) or isinstance(self, Settlement):
            return self.region_id and self.region._geo_id or None
        elif isinstance(self, Region):
            return self._geo_id
        return None

    @property
    def url_target(self):
        from travel.avia.library.python.common.models.geo import Station, Settlement

        if isinstance(self, Settlement):
            return 'city'
        elif isinstance(self, Station):
            return 'station'
        else:
            raise NotImplementedError("Urls for type %s are not supported" % self.__class__.__name__)

    def L_short_title(self, *args, **kwargs):
        return self.L_title(*args, **kwargs)

    def L_popular_title(self, *args, **kwargs):
        return self.L_title(*args, **kwargs)

    @property
    def type(self):
        from travel.avia.library.python.common.models_abstract.geo import BaseStation
        from travel.avia.library.python.common.models.geo import Settlement, Region, Country

        warnings.warn("[2016-04-21] Don't use point.type.", RaspDeprecationWarning, stacklevel=2)

        if isinstance(self, Settlement):
            return 'settlement'
        elif isinstance(self, BaseStation):
            return 'station'
        elif isinstance(self, Region):
            return 'region'
        elif isinstance(self, Country):
            return 'country'

        raise NotImplementedError('Property "type" for {} is not supported'.format(self.__class__.__name__))


class StationCodeManager(models.Manager):
    def __init__(self):
        models.Manager.__init__(self)

        self.station_cache = None
        self.code_cache = None

    def get_list_by_code(self, code):
        from travel.avia.library.python.common.models.geo import Station

        if self.station_cache is not None:
            pks = self.station_cache.get(code.lower(), [])

            return self.model.objects.in_bulk_list(pks)

        return list(Station.objects.filter(hidden=False, code_set__code=code))

    def getcodes(self, station):
        from travel.avia.library.python.common.models.geo import StationCode

        if self.code_cache is not None:
            return self.code_cache.get(station.id, {})

        return dict(
            (sc.system.code, sc.code)
            for sc in StationCode.objects.filter(station=station)
        )

    def precache(self, systems=None):
        from travel.avia.library.python.common.models.geo import CodeSystem, StationCode

        station_cache = {}
        code_cache = {}

        codesystems = dict(
            (cs.id, cs.code)
            for cs in CodeSystem.objects.all()
        )

        if not systems:
            systems = codesystems.values()

        for sc in StationCode.objects.filter(station__hidden=False, system__code__in=systems):
            key = sc.code.lower()

            station_cache.setdefault(key, []).append(sc.station_id)

            system_code = codesystems[sc.system_id]

            code_cache.setdefault(sc.station_id, {})[system_code] = sc.code

        self.station_cache = station_cache
        self.code_cache = code_cache
