# -*- coding: utf-8 -*-

"""
Эти хелперы используются в абстрактных моделях и поэтому не должны зависеть от обычных моделей
"""

import itertools
import json
import logging
from functools import wraps

from django.conf import settings
from django.db import models
from django.utils import translation
from django.utils.encoding import force_unicode, smart_str
from django.utils.functional import lazy, cached_property
from django.utils.translation import ugettext_lazy as _, ugettext

from travel.avia.library.python.common.utils.caching import CachingManager
from travel.avia.library.python.common.utils.fields import TrimmedCharField
from travel.avia.library.python.common.utils.geobase import geobase
from travel.avia.library.python.common.utils.text import normalize
from travel.avia.library.python.common.xgettext.i18n import markdbtrans

log = logging.getLogger(__name__)


class L_stops_mixin(object):
    def L_stops(self):
        if not getattr(self, 'stops_translations', None):
            return ''

        lang = translation.get_language()

        stops = json.loads(self.stops_translations)

        if lang in stops:
            return stops[lang]

        return ''


def tankerupload(tanker, model, objects, field, keyset):
    keys = {}

    field_object = getattr(model, 'L_' + field)

    if hasattr(model, 'tanker_add_{}_filter'.format(field)):
        objects = getattr(model, 'tanker_add_{}_filter'.format(field))(objects)

    for obj in objects:
        translations = {}

        field_translations = getattr(obj, 'L_' + field)

        for lang, attname, form in field_translations.by_language():
            local_language = obj.get_local_language() if hasattr(obj, 'get_local_language') else field_object.base_lang

            if lang is None:
                lang = local_language or 'ru'

            if form:
                translations[lang] = {
                    "form": form,
                }

        if translations:
            context = [
                ugettext(field_object.verbose_name),
            ]

            for context_field in model.TANKER_L_FIELDS:
                context_field_obj = getattr(model, 'L_' + context_field)
                context_field_translations = getattr(obj, 'L_' + context_field)

                context.append('%s: "%s"' % (ugettext(context_field_obj.verbose_name), context_field_translations.base_value))

            tanker_context_help_fields = getattr(model, 'TANKER_HELP_FIELDS', [])
            if tanker_context_help_fields:
                context.append(u"")

                for help_field_name in getattr(model, 'TANKER_HELP_FIELDS', []):
                    help_field = model._meta.get_field(help_field_name)
                    help_value = getattr(obj, help_field_name)

                    context.append(u'%s: "%s"' % (ugettext(help_field.verbose_name), help_value))

            key_field_name = getattr(model, 'TANKER_KEY_FIELD', 'id')
            translation_key = unicode(getattr(obj, key_field_name))

            keys[translation_key] = {
                "info": {
                    "is_plural": False,
                    "context": '\n'.join(context)
                },

                "translations": translations
            }

    tanker.upload(keyset, keys, L_field.LANGUAGES)


def tankerdownload(tanker, objects, field, keyset):
    model = objects.model
    updated = 0

    print smart_str(
        u'Downloading %s[%s]' % (
            force_unicode(keyset),
            ','.join(force_unicode(lang) for lang in L_field.LANGUAGES),
        )
    )

    keys = tanker.download(keyset, L_field.LANGUAGES)

    key_field_name = getattr(model, 'TANKER_KEY_FIELD', 'id')

    filter_kwargs = {'{}__in'.format(key_field_name): keys.keys()}
    objects = objects.filter(**filter_kwargs)

    for obj in objects:
        fields_updated = 0
        translation_key = unicode(getattr(obj, key_field_name))

        if translation_key not in keys:
            print smart_str(u'Object %s has not translation in tanker' % translation_key)

            continue

        key = keys[translation_key]

        field_translations = getattr(obj, 'L_' + field)

        for lang, attname, form in field_translations.by_language():
            if lang is None:
                continue

            override = getattr(obj, attname + '_override', False)

            if not override:
                try:
                    translated_form = key['translations'][lang]['form'].strip()
                except KeyError:
                    continue

                if translated_form and translated_form != form:
                    print smart_str("Old: [%s]" % form)
                    print smart_str("New: [%s]" % translated_form)

                    setattr(obj, attname, translated_form)

                    updated += 1
                    fields_updated += 1

        if fields_updated:
            obj.save()

    print smart_str(u'Updated %d objects' % updated)


class FalseUnicode(unicode):
    def __nonzero__(self):
        return False


class Translations(object):
    def __init__(self, field, obj):
        self.field = field
        self.obj = obj

    def __call__(self, case='nominative', lang=None, fallback=True, **kwargs):
        lang = lang or translation.get_language()
        linguistics = (
            self.field.linguistics
            if fallback else
            self.field.exact_linguistics
        )
        form = linguistics(self.obj, lang, case, **kwargs)
        marked = markdbtrans(form)

        if not form and marked:
            return FalseUnicode(marked)

        return marked

    def dict(self):
        return dict(
            (field, getattr(self.obj, field))
            for field in self.field.local_fields()
        )

    def by_language(self):
        if self.field.add_local_field:
            attname = self.field.field_name
            yield None, attname, getattr(self.obj, attname)

        for lang in self.field.LANGUAGES:
            attname = '%s_%s' % (self.field.field_name, lang)

            yield lang, attname, getattr(self.obj, attname)

    @property
    def base_value(self):
        parts = [self.field.field_name]

        if self.field.base_lang:
            parts.append(self.field.base_lang)

        return getattr(self.obj, '_'.join(parts))

    def contains(self, value, transformer=normalize):
        """
        Возвращает True если хотябы одно из локализованных значений равно value
        """
        value = transformer(value)

        for attr_name in self.field.local_fields():
            if transformer(getattr(self.obj, attr_name)) == value:
                return True

        return False

    def __repr__(self):
        return '<%s %r>' % (
            self.__class__.__name__,
            self.dict()
        )


class NoneTranslations(object):
    def __call__(self, *args, **kwargs):
        return u''

    def contains(self, *args, **kwargs):
        return False


class ConstantTranslations(object):
    def __init__(self, value):
        self.value = value

    def __call__(self, *args, **kwargs):
        return self.value

    def contains(self, value, transformer=normalize):
        return transformer(value) == transformer(self.value)


def nonzero_return(default=None):
    def decorator(func):
        @wraps(func)
        def wrapped(*args, **kwargs):
            generator = func(*args, **kwargs)
            return next(itertools.ifilter(None, generator), default)

        return wrapped

    return decorator


class L_field(object):
    LANGUAGES = settings.MODEL_LANGUAGES

    def __init__(self, verbose_name,
                 critical=False,
                 add_local_field=False,
                 local_field_critical=False,
                 local_field_unique=False,
                 base_lang=None,
                 base_field_critical=False,
                 add_override_field=False,
                 field_cls=None,
                 extra={},
                 fallback_fields=[],
                 **kwargs):
        self.verbose_name = verbose_name
        self.critical = critical
        self.add_local_field = add_local_field
        self.local_field_critical = local_field_critical
        self.local_field_unique = local_field_unique
        self.base_lang = base_lang
        self.base_field_critical = base_field_critical
        self.add_override_field = add_override_field
        self.field_cls = field_cls or self.char_field
        self.extra = extra
        self.fallback_fields = fallback_fields
        self.kwargs = kwargs

    @classmethod
    def admin_fields(cls, model, fields, group_by='field', **kwargs):
        return tuple(cls._admin_fields(model, fields, group_by, **kwargs))

    @classmethod
    def _admin_fields(cls, model, fields, group_by='field', **kwargs):
        if group_by == 'field':
            for field in fields:
                obj = getattr(model, 'L_' + field)

                if kwargs.get('show_local_field') and obj.add_local_field:
                    yield obj.field_name

                for lang in cls.LANGUAGES:
                    yield '%s_%s' % (obj.field_name, lang)

                    if obj.add_override_field:
                        yield '%s_%s_override' % (obj.field_name, lang)

                    extra_fields = obj.extra.get(lang, [])

                    for case, __ in extra_fields:
                        yield '%s_%s_%s' % (obj.field_name, lang, case)

        elif group_by == 'lang':
            for lang in cls.LANGUAGES:
                for field in fields:
                    obj = getattr(model, 'L_' + field)

                    yield '%s_%s' % (obj.field_name, lang)

                    if obj.add_override_field:
                        yield '%s_%s_override' % (obj.field_name, lang)

                    extra_fields = obj.extra.get(lang, [])

                    for case, __ in extra_fields:
                        yield '%s_%s_%s' % (obj.field_name, lang, case)

        else:
            raise ValueError(u'Unknown group type: %s' % group_by)

    @classmethod
    def search_fields(cls, model, fields):
        return list(cls._search_fields(model, fields))

    @classmethod
    def _search_fields(cls, model, fields):
        for field in fields:
            obj = getattr(model, 'L_' + field)

            if obj.add_local_field:
                yield obj.field_name

            for lang in obj.LANGUAGES:
                yield '%s_%s' % (obj.field_name, lang)

                extra_fields = obj.extra.get(lang, [])

                for case, field_object in extra_fields:
                    yield '%s_%s_%s' % (obj.field_name, lang, case)

    def local_fields(self):
        if self.add_local_field:
            yield self.field_name

        for lang in self.LANGUAGES:
            yield '%s_%s' % (self.field_name, lang)

            extra_fields = self.extra.get(lang, [])

            for case, field_object in extra_fields:
                yield '%s_%s_%s' % (self.field_name, lang, case)

    @classmethod
    def char_field(cls, verbose_name, **extra_kwargs):
        kwargs = dict(default=None, max_length=100, null=True, blank=True, db_index=False)

        kwargs.update(extra_kwargs)

        return TrimmedCharField(verbose_name, **kwargs)

    def contribute_to_class(self, cls, name):
        self.name = name
        self.model = cls

        assert name.startswith('L_')

        self.field_name = name[2:]

        # Connect myself
        setattr(cls, name, self)

        if self.add_local_field:
            kwargs = self.kwargs.copy()

            if self.local_field_critical:
                kwargs['null'] = kwargs['blank'] = False

            kwargs['unique'] = self.local_field_unique

            field = self.field_cls(self.verbose_name, **kwargs)

            cls.add_to_class(self.field_name, field)

        for lang in self.LANGUAGES:
            verbose_name = lazy(lambda verbose_name, lang: '%s (%s)' % (ugettext(verbose_name), lang), unicode)(self.verbose_name, lang)

            kwargs = self.kwargs.copy()

            if self.critical or self.base_field_critical and self.base_lang == lang:
                kwargs['null'] = kwargs['blank'] = False

            cls.add_to_class('%s_%s' % (self.field_name, lang), self.field_cls(verbose_name, **kwargs))

            if self.add_override_field:
                cls.add_to_class('%s_%s_override' % (self.field_name, lang),
                                 models.BooleanField(_(u'не обновлять из танкера (%s)' % lang),
                                                     default=False))

        for lang, extra_fields in self.extra.items():
            for case, field_object in extra_fields:
                cls.add_to_class("%s_%s_%s" % (self.field_name, lang, case), field_object)

    def exact_linguistics(self, obj, lang, case, **_kwargs):
        field = self.field_name

        if case == 'nominative':
            attname = '_'.join([field, lang])
        else:
            attname = '_'.join([field, lang, case])

        return getattr(obj, attname, None)

    @nonzero_return(default=u'')
    def linguistics(self, obj, lang, case, field_fallback=True, **kwargs):
        L_fields = [self]

        if field_fallback:
            for field_name in self.fallback_fields:
                field = getattr(self.model, field_name)

                assert isinstance(field, L_field), \
                    "fallback_fields should reference L_field fields"

                L_fields.append(field)

        @nonzero_return()
        def field_fallback_linguistics(*args):
            for field in L_fields:
                yield field.exact_linguistics(obj, *args, **kwargs)

        yield field_fallback_linguistics(lang, case)

        for fallback_lang, fallback_case in settings.LANGUAGE_CASE_FALLBACKS.get((lang, case), []):
            yield field_fallback_linguistics(fallback_lang, fallback_case)

        if case != 'nominative':
            yield field_fallback_linguistics(lang, 'nominative')

        for fallback_lang in settings.LANGUAGE_FALLBACKS.get(lang, []):
            yield field_fallback_linguistics(fallback_lang, 'nominative')

        for field in L_fields:
            if field.add_local_field:
                yield getattr(obj, field.field_name)

        if self.base_lang:
            yield self.exact_linguistics(obj, self.base_lang, 'nominative')

    def __get__(self, instance, instance_type=None):
        if instance is None:
            return self

        return Translations(self, instance)

    def __set__(self, instance, value):
        if instance is None:
            raise AttributeError(u"%s must be accessed via instance" % self.name)

        if isinstance(value, Translations):
            for field in self.local_fields():
                field_value = getattr(value.obj, field)

                setattr(instance, field, field_value)

        else:
            raise ValueError("%r assignment is not supported" % type(value))


class new_L_field(L_field):
    def exact_linguistics(self, obj, lang, case, **_kwargs):
        if case == 'preposition_v_vo_na':
            return super(new_L_field, self).exact_linguistics(
                obj, lang, case, **_kwargs
            )

        kwargs = {'lang': lang}
        if case is not None:
            kwargs['case'] = case

        return getattr(
            obj, 'new_L_' + self.field_name
        ).translation(**kwargs)


class new_Geobase_L_title(new_L_field):
    def contribute_to_class(self, cls, name):
        assert name == 'L_title'
        return super(new_Geobase_L_title, self).contribute_to_class(cls, name)

    def exact_linguistics(self, obj, lang, case):
        super_form = super(new_Geobase_L_title, self).exact_linguistics(
            obj, lang, case
        )
        if super_form:
            return super_form
        return self.exact_geobase_linguistic(obj, lang, case)

    @staticmethod
    def exact_geobase_linguistic(obj, lang, case):
        if obj._geo_id and geobase:
            try:
                linguistics = geobase.linguistics(obj._geo_id, str(lang))
            except RuntimeError:
                log.warning(
                    u'Не удалось получить наименования для %s, язык %s',
                    obj._geo_id, lang
                )
            else:
                if linguistics:
                    try:
                        return getattr(linguistics, case)
                    except AttributeError:
                        pass


class RouteLTitle(object):
    """
    Позволяет вычислять title объекта по последовательности точек (станций, городов, стран).

    В модели должен быть объявлен как RouteLTitle.L_field()
    """

    LANGUAGES = ['ru', 'tr', 'uk']

    prefetched_points = None

    def __init__(self, instance):
        self.instance = instance

    @cached_property
    def title_dict(self):
        return self.extract_title_dict(self.instance.title_common)

    @classmethod
    def get_base_language(cls, obj):
        return 'ru'

    @classmethod
    def tankerupload(cls, tanker, t_type_code):
        from travel.avia.library.python.common.models.schedule import Route, RThread

        keys = {}

        routes = Route.objects.filter(is_manual_title=True)
        threads = RThread.objects.filter(is_manual_title=True)
        if t_type_code != 'all':
            routes = routes.filter(t_type__code=t_type_code)
            threads = threads.filter(route__t_type__code=t_type_code)

        objects = itertools.chain(routes, threads)

        for obj in objects:
            key = obj.title

            translations = {}

            base_language = cls.get_base_language(obj)

            for lang in cls.LANGUAGES:
                if lang == base_language:
                    form = obj.title
                    author = None
                else:
                    form = getattr(obj, 'title_%s' % lang)
                    author = getattr(obj, 'title_%s_author' % lang)

                if form:
                    translations[lang] = {
                        "form": form,
                    }

                    if author:
                        translations[lang]["author"] = author

            keys[key] = {
                "info": {'is_plural': False},
                "translations": translations
            }

        tanker.upload('route_titles', keys, cls.LANGUAGES)

    @classmethod
    def tankerdownload(cls, tanker):
        from travel.avia.library.python.common.models.schedule import Route, RThread

        keys = tanker.download('route_titles', cls.LANGUAGES)

        objects = itertools.chain(Route.objects.filter(is_manual_title=True),
                                  RThread.objects.filter(is_manual_title=True))

        for obj in objects:
            key = obj.title

            base_language = cls.get_base_language(obj)

            for lang in cls.LANGUAGES:
                if lang == base_language:
                    continue

                if key in keys:
                    translation = keys[key]["translations"][lang]

                    form = translation["form"]

                    if form:
                        setattr(obj, 'title_%s' % lang, form)
                        setattr(obj, 'title_%s_author' % lang, translation["author"])

            obj.save()

    def __call__(self, lang=None, short=False, popular=False):
        from travel.avia.library.python.common.utils.title_generator import TitleGeneratorError, TitleGenerator

        instance = self.instance

        if lang is None:
            lang = translation.get_language()

        if getattr(instance, 'is_manual_title', False):
            title = getattr(instance, 'title_{}'.format(lang), instance.title)

            if title:
                return title

        base_language = self.get_base_language(instance)

        if self.title_dict and (lang != base_language or popular):
            try:
                return TitleGenerator.L_title(
                    self.title_dict, self.prefetched_points,
                    short=short, lang=lang, popular=popular
                )
            except TitleGeneratorError:
                log.exception(u'Ошибка генерации названия %r', self.title_dict)

        if short:
            return instance.title_short or instance.title

        return instance.title

    @classmethod
    def extract_title_dict(cls, title_common):
        if not title_common:
            return

        try:
            return json.loads(title_common)
        except ValueError:
            # Поддержка старого title_common
            return {
                'type': 'default',
                'title_parts': title_common.split(u'_')
            }

    @property
    def point_keys(self):
        from travel.avia.library.python.common.utils.title_generator import TitleGenerator

        if self.title_dict:
            return TitleGenerator.extract_point_keys(self.title_dict)

        return ()

    @classmethod
    def fetch(cls, fields):
        from travel.avia.library.python.common.models.geo import Point

        point_keys = set()

        for field in fields:
            point_keys.update(field.point_keys)

        points = Point.in_bulk(point_keys)

        for field in fields:
            field.prefetched_points = points

    class L_field(object):

        def contribute_to_class(self, cls, name):
            self.name = name
            setattr(cls, name, self)

        def __get__(self, instance, _instance_type=None):
            if instance is None:
                return self

            field = RouteLTitle(instance)

            # кэшируем поле в инстансе
            setattr(instance, self.name, field)

            return field


class AbstractTranslate(models.Model):
    """
    Перевод значения поля, но в отличае от L_field хранит переведенное значение в отдельной таблице.

    Это может быть полезно в нескольких случаях:

    1) Значение поля объекта повторяется для множества объектов.
    2) Значение поля одинаково для нескольких моделей

    Хранение переводов в отдельной таблице позволяет уменьшить кол-во ручной работы по переводу


    Для работы с полем необходимо описать модель для хранения переводов следующим образом:

    class DirectionTranslate(AbstractTranslate):
        '''
        Переводы названий направлений
        '''

        @classmethod
        def get_keys(cls):
            # Доступные ключи для перевода по объектам модели

        class Meta:
            ordering = ('value',)
            verbose_name = _(u'перевод названия направления')
            verbose_name_plural = _(u'переводы названий направлений')
            app_label = 'www'

    В таблице переводов для колонки value должен быть выставлен collation в utf8_bin (ALTER TABLE table_name
    MODIFY value VARCHAR(100) CHARACTER SET utf8 COLLATE utf8_bin)


    Далее необходимо описать поле с переводами в моделе

    L_title = DirectionTranslate.get_L_method(key_field='title')

    где title - название поля содержащее ключи перевода


    Таблицу переводов необходимо заполнять скриптом fill_translates.py

    Для выгрузки и загрузки в танкер нужно пользоваться dbtanker.py
    """

    TANKER_KEY_FIELD = 'value'
    TANKER_L_FIELDS = ['value']

    L_value = L_field(verbose_name=_(u'значение'), add_local_field=True, local_field_unique=True)
    objects = CachingManager(keys=['value'], use_get_fallback=False)

    @classmethod
    def get(cls, value, lang=None):
        value = value and value.strip()

        if value:
            try:
                translate = cls.get_translate(value).L_value(lang=lang)

                if translate:
                    return translate

            except cls.DoesNotExist:
                pass

        return markdbtrans(value)

    @classmethod
    def get_translate(cls, value):
        return cls.objects.get(value=value)

    @classmethod
    def get_L_method(cls, key_field):
        @property
        def method(self):
            key = getattr(self, key_field)

            try:
                return cls.get_translate(key).L_value

            except cls.DoesNotExist:
                return ConstantTranslations(key)

        return method

    @classmethod
    def update(cls):
        """
        Производит обновление доступных ключей
        """

        cls.update_values(cls.get_keys())

    @classmethod
    def get_keys(cls):
        """
        Строит список (set) доступных ключей для перевода по объектам модели
        """

        raise NotImplementedError("get_keys method should be overridden")

    @classmethod
    def update_values(cls, values):
        """
        Добавляет новые значения для перевода

        values - set ключей для перевода
        """

        current = set(cls.objects.values_list('value', flat=True))

        new_values = values - current - set([None, ''])

        for v in new_values:
            cls(value=v, value_ru=v).save()

    class Meta:
        abstract = True
