# -*- coding: utf-8 -*-
import calendar
import json
import logging
import math
import time
from collections import defaultdict
from datetime import datetime  # noqa

import pytz
from abc import ABCMeta, abstractmethod
from django.conf import settings
from requests import Session
from six import add_metaclass
from typing import Optional, Dict  # noqa

from travel.avia.library.python.common.utils.requests_utils import requests_retry_session

log = logging.getLogger(__name__)


@add_metaclass(ABCMeta)
class ISaasIndex(object):
    @abstractmethod
    def index(self, key, doc, labels=None, expires_at=None, realtime=True,
              timeout=None, options=None):
        # type: (str, Dict[str, str], Dict[str, str], Optional[datetime], bool, Optional[float], Optional[dict]) -> None
        """
        :param key: уникальный ключ документа, как в key-value хранилище.
        :param doc: колонки документа и их значения.
        :param labels: метки для дополнительной индексации. У каждой метки есть имя и значение.
        Потом документы можно искать по значению метки с данным именем. У одной метки может быть
        несколько значений, но это пока не реализовано.
        :param expires_at: aware или naive utc время, до которого документ должен быть доступен для поиска.
        При expires_at=None документ хранится вечно.
        :param realtime: сделать документ доступным для поиска не асинхронно, а сразу же. Запрос работает дольше.
        :param timeout: сколько секунд мы можем ждать ответ.
        :param options: опции документа.
        :return:
        """
        pass

    def delete(self, key, realtime=True, timeout=None):
        # type: (str, bool, Optional[float]) -> None
        pass


class SaasIndex(ISaasIndex):
    def __init__(self, index_host, service_key, retry_5xx=False):
        self._session = Session()
        if retry_5xx:
            self._session = requests_retry_session(
                backoff_factor=0.03,
                status_forcelist=(504, 598),
                session=self._session
            )
        self._index_host = index_host
        self._service_key = service_key

    def index(self, key, doc, labels=None, expires_at=None, realtime=True,
              timeout=None, options=None):
        options = options or {}
        if expires_at:
            if expires_at.tzinfo:
                expires_at = expires_at.astimezone(pytz.UTC)
            options['deadline_min_utc'] = int(math.ceil(
                calendar.timegm(expires_at.timetuple()) / 60
            ))

        files = {}

        # SaaS разрешает одноимённые колонки и ключи
        document = defaultdict(list)

        for index, (column, value) in enumerate(doc.iteritems()):
            attachment_name = '$attachment{}'.format(index)

            document[column].append({
                'type': '#p',
                'value': attachment_name,
            })

            files[attachment_name] = value

        for label, value in (labels or {}).iteritems():
            document[label].append({
                'type': '#k',
                'value': value,
            })

        files['json_message'] = json.dumps({
            'docs': [dict(
                document,
                url=str(key),
                options=options,
            )],
            'action': 'modify',
            'prefix': 0,
        })

        response = self._session.post(
            self._url(),
            params={'realtime': 'true'} if realtime else None,
            timeout=timeout,
            files=files,
        )
        if response.status_code != 200:
            log.warning(
                'Index error for key %r: %s - %s. json_message: %r',
                key, response.status_code, response.url, files['json_message']
            )

    def delete(self, key, realtime=True, timeout=None):
        response = self._session.post(
            self._url(),
            params={'realtime': 'true'} if realtime else None,
            files={
                'json_message': json.dumps({
                    'docs': [{'url': key}],
                    'action': 'delete',
                    'prefix': 0,
                })
            },
            timeout=timeout,
        )
        if not response.ok:
            log.error('Deletion error for key %r: %r', key, response)

    def _url(self):
        return '{host}/service/{service_key}'.format(
            host=self._index_host,
            service_key=self._service_key,
        )


class LoggingIndex(ISaasIndex):
    def __init__(self, index):
        # type: (ISaasIndex) -> None
        self._origin = index

    def index(self, key, *args, **kwargs):
        ts = time.time()
        try:
            return self._origin.index(key, *args, **kwargs)
        finally:
            log.info('Indexing %s has taken %f seconds', key, time.time() - ts)

    def delete(self, key, *args, **kwargs):
        ts = time.time()
        try:
            return self._origin.delete(key, *args, **kwargs)
        finally:
            log.info('Deleting %s has taken %f seconds', key, time.time() - ts)


saas_index = LoggingIndex(SaasIndex(
    index_host=settings.SAAS_INDEX_HOST,
    service_key=settings.SAAS_SERVICE_KEY,
))
