# -*- coding: utf-8 -*-
import json
import logging
from datetime import datetime, date

from typing import Optional, Dict, Iterable  # noqa

from .index import ISaasIndex, saas_index  # noqa
from .search import ISaasSearch, saas_search  # noqa

log = logging.getLogger(__name__)


class DatetimeEncoder(json.JSONEncoder):
    def default(self, obj):
        if isinstance(obj, (datetime, date)):
            return obj.isoformat()
        return super(DatetimeEncoder, self).default(obj)


def _serialized(doc):
    return {
        column: DatetimeEncoder().encode(obj)
        for column, obj in doc.iteritems()
    }


def _deserialized(json_doc):
    return {
        column: data if column.startswith('_SK_') else json.loads(data)
        for column, data in json_doc.iteritems()
    }


class JsonSaas(object):
    def __init__(self, index, search):
        # type: (ISaasIndex, ISaasSearch) -> None
        self._index = index
        self._search = search

    def index(self, key, doc, labels=None, expires_at=None, realtime=True, timeout=None):
        # type: (str, Dict[str, object], Dict[str, str], Optional[datetime], bool, Optional[float]) -> None
        return self._index.index(
            key=key,
            doc=_serialized(doc),
            labels=labels,
            expires_at=expires_at,
            realtime=realtime,
            timeout=timeout,
        )

    def search(self, keys, columns=None, label=None, low_priority=False, timeout=None):
        # type: (Iterable[str], Iterable[str], Optional[str], bool, Optional[float]) -> Dict[str, Dict[str, object]]
        docs = self._search.search(
            keys=keys,
            columns=columns,
            label=label,
            low_priority=low_priority,
            timeout=timeout,
        )
        return {
            key: _deserialized(doc)
            for key, doc in docs.iteritems()
        }

    def delete(self, key, realtime=True, timeout=None):
        # type: (str, bool, Optional[float]) -> None
        return self._index.delete(key, realtime=realtime, timeout=timeout)


json_saas = JsonSaas(
    index=saas_index,
    search=saas_search,
)
