# -*- coding: utf-8 -*-
import logging
import time

from abc import ABCMeta, abstractmethod
from django.conf import settings
from requests import Session
from six import add_metaclass
from typing import Optional, Dict, Iterable  # noqa

from .meta_pb2 import TReport
from travel.avia.library.python.common.utils.requests_utils import requests_retry_session

log = logging.getLogger(__name__)


@add_metaclass(ABCMeta)
class ISaasSearch(object):
    @abstractmethod
    def search(self, keys, columns=None, label=None, low_priority=False, timeout=None):
        # type: (Iterable[str], Iterable[str], Optional[str], bool, Optional[float]) -> Dict[str, Dict[str, str]]
        """
        :param keys: значения ключей, по которым нужно искать. Будут найдены все документы,
        ключ которых есть в списке. Если указан label, ключами являются значения метки.
        :param columns: колонки документов, которые нужно вернуть. При columns=None возвращаются
        все колонки документа; если при этом поиск был по метке, вернётся значение метки
        с префиксом _SK_, чтобы избежать коллизий.
        :param label: метка для поиска. Если не указана, поиск идёт по основным ключам документов.
        :param low_priority: запросы с low_priority=True обрабатываются отдельным небольшим пулом потоков и не
        мешают запросам с low_priority=False. Сделано на случай, если где-то важна скорость поиска.
        :param timeout: сколько секунд ждать ответ.
        :return: Возвращается список найденных документов. Каждый документ лежит в ответе по своему ключу
        и состоит из колонок-значений.
        """
        pass


class SaasSearch(ISaasSearch):
    def __init__(self, search_host, service_name, retry_5xx=False):
        self._session = Session()
        if retry_5xx:
            self._session = requests_retry_session(
                backoff_factor=0.03,
                status_forcelist=(504, 598),
                session=self._session
            )
        self._search_host = search_host
        self._service = service_name

    def search(self, keys, columns=None, label=None, low_priority=False, timeout=None):
        keys = tuple(set(keys))
        if not keys:
            return {}

        params = {
            'service': self._service,
            'text': ','.join(keys),
            'ms': 'proto',
        }
        if label is not None:
            params['key_name'] = label
        if low_priority:
            params['pron'] = 'longsearch'
        if timeout:
            params['timeout'] = int(1000000 * timeout)  # микросекунды

        response = self._session.get(
            self._search_host,
            params=params.items() + [('gta', col) for col in columns or ()],
            timeout=timeout,
        )

        if response.status_code != 200:
            log.info(
                "Couldn't get a result for %s[%s]: %r - %s",
                label or 'PRIMARY', keys, response.status_code, response.url
            )
            return {}

        allowed = set(columns).__contains__ if columns else lambda x: True

        return {
            doc.Url: {
                attr.Key: attr.Value
                for attr in doc.ArchiveInfo.GtaRelatedAttribute
                if allowed(attr.Key)
            }
            for doc in (
                group.Document[0]
                for group in TReport.FromString(response.content).Grouping[0].Group
            )
        }


class LoggingSearch(ISaasSearch):
    def __init__(self, search):
        # type: (ISaasSearch) -> None
        self._origin = search

    def search(self, keys, *args, **kwargs):
        ts = time.time()
        try:
            return self._origin.search(keys, *args, **kwargs)
        finally:
            log.info('Searching for %s has taken %f secs', keys, time.time() - ts)


saas_search = LoggingSearch(SaasSearch(
    search_host=settings.SAAS_SEARCH_HOST,
    service_name=settings.SAAS_SERVICE_NAME,
))
