# -*- coding: utf-8 -*-
import os.path
import socket

from travel.avia.library.python.common.mdb.mysql import get_replica_info
from travel.avia.library.python.common.utils.dcutils import dc_cache, dc_cache_settings
from travel.avia.library.python.common.settings.utils import get_caches_config_by_group

YANDEX_ENVIRONMENT_TYPE_FILE = '/etc/yandex/environment.type'
YANDEX_ENVIRONMENT_NAME_FILE = '/etc/yandex/environment.name'


class EnvironmentSettings(object):
    registered_configs = {}

    @staticmethod
    def read_string_from_file(filepath):
        if os.path.exists(filepath):
            with open(filepath, 'r') as f:
                return f.read().strip()

    @classmethod
    def apply(cls, settings, conductor=None, use_master_db=False):
        settings['USE_MASTER_DB'] = use_master_db

        cls.conductor = conductor

        cls.yandex_environment_type = settings.get(
            'YANDEX_ENVIRONMENT_TYPE',
            cls.read_string_from_file(YANDEX_ENVIRONMENT_TYPE_FILE)
        )

        cls.yandex_environment_name = settings.get(
            'YANDEX_ENVIRONMENT_NAME',
            cls.read_string_from_file(YANDEX_ENVIRONMENT_NAME_FILE)
        )

        settings['YANDEX_ENVIRONMENT_TYPE'] = cls.yandex_environment_type
        settings['YANDEX_ENVIRONMENT_NAME'] = cls.yandex_environment_name

        for db in ('default', 'writable'):
            settings['DATABASES'][db]['NAME'] = os.getenv('AVIA_MYSQL_DATABASE', 'avia')
            settings['DATABASES'][db]['USER'] = os.getenv('AVIA_MYSQL_USER', 'rasp')
            settings['DATABASES'][db]['PASSWORD'] = os.getenv('AVIA_MYSQL_PASSWORD', '')

        settings['BLACKBOX_URL'] = 'http://blackbox-mimino.yandex.net/blackbox'
        settings['CACHES'].update({
            'replica_set': {
                'BACKEND': 'django.core.cache.backends.locmem.LocMemCache',
                'timeout': 60,
            }
        })

        settings['CACHES'].update(dc_cache_settings(settings.get('CLUSTER_CACHE_HOSTS') or []))

        if cls.yandex_environment_type == 'sandbox_testing':
            cls.apply_sandbox_testing(settings)
            settings['APPLIED_CONFIG'] = 'sandbox_testing'

        elif cls.yandex_environment_type == 'sandbox_production':
            cls.apply_sandbox_production(settings)
            settings['APPLIED_CONFIG'] = 'sandbox_production'

        elif cls.yandex_environment_type == 'stress':
            cls.apply_stress(settings)
            settings['APPLIED_CONFIG'] = 'stress'

        elif cls.yandex_environment_type == 'testing':
            if cls.yandex_environment_name == 'localhost':
                cls.apply_openstack(settings)
                settings['APPLIED_CONFIG'] = 'openstack'
            else:
                cls.apply_testing(settings)
                settings['APPLIED_CONFIG'] = 'testing'

        elif cls.yandex_environment_type == 'development':
            cls.apply_development(settings)
            settings['APPLIED_CONFIG'] = 'development'

        elif cls.yandex_environment_type == 'production':
            cls.apply_production(settings)
            settings['APPLIED_CONFIG'] = 'production'

        else:
            cls.apply_other(settings)
            settings['APPLIED_CONFIG'] = 'other'

        return cls

    @classmethod
    def register_config(cls, environtment_name):
        """
        Пример использования

        @EnvironmentSettings.register_config('testing')
        def configure_testing(environment_settings, settings)):
            settings['TICKETS_URL'] = 'http://daemon.lepus.avia.yandex.net'

        EnvironmentSettings.apply(globals(), conductor)
        """

        def registrator(configurator_fn):
            cls.registered_configs.setdefault(environtment_name, []).append(configurator_fn)

            return configurator_fn

        return registrator

    @classmethod
    def configure(cls, environment_name, settings):
        for configurator_fn in cls.registered_configs.get(environment_name, []):
            configurator_fn(cls, settings)

    @classmethod
    def apply_stress(cls, settings):
        replicainfo = cls.conductor.get_mysql_replicainfo(
            'avia_load_db_mysql_main', settings.get('CURRENT_DC')
        )
        settings['DATABASES']['default']['REPLICAS'] = replicainfo.replicas

        settings['DATABASES']['writable']['REPLICAS'] = replicainfo.replicas
        settings['DATABASES']['writable']['HOST'] = replicainfo.master

        settings['CACHES'].update(get_caches_config_by_group(
            'rasp_load_mcrouter', cls.conductor, settings.get('CURRENT_DC')
        ))

        settings['HAS_VALID_YANDEXUID_ALWAYS_TRUE'] = True
        settings['CURRENCY_RATES_URL'] = 'http://units-converter-r1.n.yandex-team.ru/converter-rasp'

        settings['BLACKBOX_URL'] = 'http://pass-stress-na1.sezam.yandex.net'
        settings['GEO_SNIPPETER_URL'] = 'http://snippetter01g.load.maps.yandex.net'

        hostname = socket.gethostname()
        settings['PATHFINDER_URL'] = 'http://targetpathfinder.rasp.{}/search'.format(hostname)

        cls.configure('stress', settings)

    @classmethod
    def apply_other(cls, settings):
        raise Exception("Environment type %r not supported" % cls.yandex_environment_type)

    @classmethod
    def apply_testing(cls, settings):
        if settings.get('MYSQL_CLUSTER_ID'):
            replicainfo = get_replica_info(settings, settings.get('CURRENT_DC'))
        else:
            replicainfo = cls.conductor.get_groups_mysql_replicainfo(
                'avia_test_db_mysql_main', 'avia_test_db_mysql_main_master', settings.get('CURRENT_DC')
            )

        settings['DATABASES']['default']['REPLICAS'] = replicainfo.replicas
        if settings.get('USE_MASTER_DB'):
            settings['DATABASES']['default']['USE_MASTER_DB'] = True
            settings['DATABASES']['default']['HOST'] = replicainfo.master

        settings['DATABASES']['writable']['REPLICAS'] = replicainfo.replicas
        settings['DATABASES']['writable']['USE_MASTER_DB'] = True
        settings['DATABASES']['writable']['HOST'] = replicainfo.master

        settings['MEDIA_URL'] = '//admin.avia.tst.yandex-team.ru/media/'

        settings['TICKET_HOST'] = 'avia.lepus.yandex.ru'
        settings['TICKET_URL'] = 'https://avia.lepus.yandex.ru/'

        settings['PATHFINDER_URL'] = 'http://targetpathfinder.lepus.rasp.yandex.net/search'

        settings['GEO_SNIPPETER_URL'] = 'http://snippetter01h.tst.maps.yandex.ru'
        settings['PARTNERKA_URL'] = 'https://testing-avia-partners.common.yandex.ru'

        cls.configure('testing', settings)

    @classmethod
    def apply_openstack(cls, settings):
        for db in ('default', 'writable'):
            settings['DATABASES'][db]['HOST'] = os.getenv('AVIA_MYSQL_HOST', 'localhost')
            settings['DATABASES'][db]['PASSWORD'] = os.getenv('AVIA_MYSQL_PASSWORD', 'rasp')

        caches = {
            'default': dc_cache(['localhost'])
        }

        settings['CACHES'].update(caches)

        cls.configure('openstack', settings)

    @classmethod
    def apply_development(cls, settings):
        for db in ('default', 'writable'):
            settings['DATABASES'][db]['HOST'] = os.getenv('AVIA_MYSQL_HOST', 'localhost')
            settings['DATABASES'][db]['PASSWORD'] = os.getenv('AVIA_MYSQL_PASSWORD', 'rasp')

        caches = {
            'default': dc_cache(['localhost'])
        }

        settings['CACHES'].update(caches)

        cls.configure('development', settings)

    @classmethod
    def apply_production(cls, settings):
        if settings.get('MYSQL_CLUSTER_ID'):
            replicainfo = get_replica_info(settings, settings.get('CURRENT_DC'))
        else:
            replicainfo = cls.conductor.get_groups_mysql_replicainfo(
                'avia_db_mysql_main', 'avia_db_mysql_main_master', settings.get('CURRENT_DC')
            )

        settings['DATABASES']['default']['REPLICAS'] = replicainfo.replicas
        if settings.get('USE_MASTER_DB'):
            settings['DATABASES']['default']['USE_MASTER_DB'] = True
            settings['DATABASES']['default']['HOST'] = replicainfo.master

        settings['DATABASES']['writable']['REPLICAS'] = replicainfo.replicas
        settings['DATABASES']['writable']['USE_MASTER_DB'] = True
        settings['DATABASES']['writable']['HOST'] = replicainfo.master

        settings['BLACKBOX_URL'] = 'http://blackbox.yandex.net/blackbox'

        settings['MEDIA_URL'] = '//yastatic.net/rasp/media/'
        settings['GEO_SNIPPETER_URL'] = 'http://snippetter.maps.yandex.net'

        settings['CURRENCY_RATES_URL'] = 'http://units-converter.yandex.net:11000/converter-rasp'

        settings['PARTNERKA_URL'] = 'https://partners.avia.yandex.ru'

        cls.configure('production', settings)

    @classmethod
    def apply_sandbox_testing(cls, settings):
        if settings.get('MYSQL_CLUSTER_ID'):
            replicainfo = get_replica_info(settings, settings.get('CURRENT_DC'))
        else:
            replicainfo = cls.conductor.get_groups_mysql_replicainfo(
                'avia_test_db_mysql_main', 'avia_test_db_mysql_main_master', settings.get('CURRENT_DC')
            )

        settings['DATABASES']['default']['REPLICAS'] = replicainfo.replicas
        if settings.get('USE_MASTER_DB'):
            settings['DATABASES']['default']['USE_MASTER_DB'] = True
            settings['DATABASES']['default']['HOST'] = replicainfo.master

        settings['DATABASES']['writable']['REPLICAS'] = replicainfo.replicas
        settings['DATABASES']['writable']['USE_MASTER_DB'] = True
        settings['DATABASES']['writable']['HOST'] = replicainfo.master

        settings['MEDIA_URL'] = '//admin.avia.tst.yandex-team.ru/media/'

        settings['TICKET_HOST'] = 'avia.lepus.yandex.ru'
        settings['TICKET_URL'] = 'https://avia.lepus.yandex.ru/'

        settings['PATHFINDER_URL'] = 'http://targetpathfinder.lepus.rasp.yandex.net/search'

        settings['GEO_SNIPPETER_URL'] = 'http://snippetter01h.tst.maps.yandex.ru'
        settings['PARTNERKA_URL'] = 'https://testing-avia-partners.common.yandex.ru'

        cls.configure('testing', settings)

    @classmethod
    def apply_sandbox_production(cls, settings):
        if settings.get('MYSQL_CLUSTER_ID'):
            replicainfo = get_replica_info(settings, settings.get('CURRENT_DC'))
        else:
            replicainfo = cls.conductor.get_groups_mysql_replicainfo(
                'avia_db_mysql_main', 'avia_db_mysql_main_master', settings.get('CURRENT_DC')
            )

        settings['DATABASES']['default']['REPLICAS'] = replicainfo.replicas
        if settings.get('USE_MASTER_DB'):
            settings['DATABASES']['default']['USE_MASTER_DB'] = True
            settings['DATABASES']['default']['HOST'] = replicainfo.master

        settings['DATABASES']['writable']['REPLICAS'] = replicainfo.replicas
        settings['DATABASES']['writable']['USE_MASTER_DB'] = True
        settings['DATABASES']['writable']['HOST'] = replicainfo.master

        settings['BLACKBOX_URL'] = 'http://blackbox.yandex.net/blackbox'

        settings['MEDIA_URL'] = '//yastatic.net/rasp/media/'
        settings['GEO_SNIPPETER_URL'] = 'http://snippetter.maps.yandex.net'

        settings['CURRENCY_RATES_URL'] = 'http://units-converter.yandex.net:11000/converter-rasp'

        settings['PARTNERKA_URL'] = 'https://partners.avia.yandex.ru'

        cls.configure('sandbox_production', settings)
