# -*- coding: utf-8 -*-

import copy
import logging
import os
import requests
import urlparse

from time import time as clock_time
from django.conf import settings

from travel.avia.library.python.common.utils.marketstat import DsvSimpleLog

antirobot_log = logging.getLogger(__name__)
antirobot_yt_log = DsvSimpleLog(
    os.path.join(settings.LOG_PATH, 'yt/antirobot.log')
)


class AntirobotApplication(object):
    def __init__(self, app):
        self.app = app
        self.call_time = None

    @property
    def exec_time(self):
        return '%sms' % int((clock_time() - self.call_time) * 1000)

    def fix_x_forwarded_for(self, header):
        ips = []
        for h in [hs.strip() for hs in header.split(',')]:
            ips.append(h[7:] if h.startswith('::ffff:') else h)

        return ', '.join(ips)

    def make_headers(self, environ):
        headers = {}

        for gunicorn_header in environ:
            if gunicorn_header.startswith('HTTP_'):
                header = gunicorn_header.replace('HTTP_', '', 1).replace('_', '-')
                headers[header] = environ[gunicorn_header]

        headers['X-Antirobot-Service-Y'] = getattr(settings, 'ANTIROBOT_SERVICE', 'avia')
        headers['X-Yandex-HTTPS'] = 'yes'

        if 'X-FORWARDED-FOR' in headers:
            headers['X-Forwarded-For-Y'] = self.fix_x_forwarded_for(headers['X-FORWARDED-FOR'])

        if 'HOST' in headers:
            headers['X-Host-Y'] = headers['HOST']

        return headers

    def make_dsv_data(self, is_robot, status_code, location=None):
        dsv_data = {
            'exec_time': self.exec_time,
            'is_robot': "1" if is_robot else "0",
            'status_code': status_code,
            'http_x_real_ip': self.http_x_real_ip,
            'request_method': self.request_method,
            'uri': self.uri,
            'tskv_format': 'rasp-antirobot-log',
        }
        return dsv_data

    def register_captcha(self, is_robot, status_code, location):
        antirobot_log.info(
            '%s Captcha 302 %s %s', self.exec_time, location, self.msg_tail
        )

        antirobot_yt_log.log(self.make_dsv_data(is_robot, status_code, location))

    def register_robot(self, is_robot, status_code):
        antirobot_log.info(
            '%s Robot 403 %s', self.exec_time, self.msg_tail
        )

        antirobot_yt_log.log(self.make_dsv_data(is_robot, status_code))

    def register_pass(self, is_robot, status_code):
        antirobot_log.info(
            '%s Pass %s %s', self.exec_time, status_code, self.msg_tail
        )

        antirobot_yt_log.log(self.make_dsv_data(is_robot, status_code))

    def register_error(self, e):
        antirobot_log.warning('%s Error query antirobot %s %r', self.exec_time, self.msg_tail, e)

    def __call__(self, environ, start_response):
        qs = environ['QUERY_STRING']
        headers = self.make_headers(environ)

        self.call_time = clock_time()
        self.uri = '%s?%s' % (environ['PATH_INFO'], qs) if qs else environ['PATH_INFO']
        self.http_x_real_ip = environ['HTTP_X_REAL_IP']
        self.request_method = environ['REQUEST_METHOD']
        self.msg_tail = '%s %s %s' % (
            self.http_x_real_ip, self.request_method, self.uri
        )

        try:
            robo_check = RoboCheck(environ, self.uri, headers)
            is_robot = robo_check.is_robot()
            status_code = robo_check.status_code

        except Exception as e:
            self.register_error(e)

        else:
            if is_robot:
                if status_code == 302:
                    location = robo_check.redirect_location

                    self.register_captcha(is_robot, status_code, location)

                    start_response('302 Redirect', [('Location', location)])

                    # В случае редиректа сразу отдаём пустое тело ответа
                    return []

                elif robo_check.status_code == 403:
                    self.register_robot(is_robot, status_code)

                    start_response('200 OK', [('Content-Type', 'text/html')])

                    return [robo_check.http_content]

            self.register_pass(is_robot, status_code)

        return self.app(environ, start_response)


class RoboCheck(object):
    def __init__(self, environ, uri, headers):
        self.environ = environ
        self.uri = uri
        self.headers = headers
        self.r = None

    @property
    def redirect_location(self):
        url_parts = list(urlparse.urlparse(self.r.headers['Location']))
        host_parts = url_parts[1].split(':')
        url_parts[1] = host_parts[0]

        return urlparse.urlunparse(url_parts)

    @property
    def http_content(self):
        return self.r.content

    @property
    def status_code(self):
        return self.r.status_code if self.r is not None else None

    def is_robot(self):
        http_method = self.environ['REQUEST_METHOD']
        url = 'http://antirobot.yandex.ru%s' % self.uri

        timeout = getattr(settings, 'ANTIROBOT_TIMEOUT', 0.1)

        if http_method == 'GET':
            self.r = requests.get(
                url=url,
                headers=self.headers,
                timeout=timeout,
                allow_redirects=False
            )

        elif http_method == 'POST':
            self.r = requests.post(
                url=url,
                headers=self.headers,
                timeout=timeout,
                data=copy.copy(self.environ['wsgi.input']).read(),
                allow_redirects=False
            )

        else:

            return None

        if self.r.status_code in [302, 403]:
            # Подозрение на робота или забаненный
            return True

        return False
