# -*- coding: utf-8 -*-
from __future__ import absolute_import

import logging
from functools import wraps
from threading import RLock
from thread import start_new_thread

from django.conf import settings
from django.core.cache import cache, caches
from django.utils.translation import get_language
from retrying import retry

from travel.avia.library.python.common.precache.manager import PrecachingManager as CachingManager  # noqa
from travel.avia.library.python.common.utils.dcutils import dc_cache_name
from travel.avia.library.python.common.utils.mysql_switcher.signals import data_updated
from travel.avia.library.python.common.utils.threadutils import run_in_thread


log = logging.getLogger(__name__)


def cache_until_switch(func):
    u"""
    Кеширует результаты выполнения функции, обновляет результаты если сменилась база.
    Все аргументы должны быть hashable.
    """
    results = {}

    def clean_cache(**kwargs):
        for k in results.keys():
            del results[k]
    data_updated.connect(clean_cache, weak=False)

    @wraps(func)
    def new_func(*args, **kwargs):
        key = args + tuple([(k, v) for k, v in kwargs.items()])
        try:
            return results[key]
        except KeyError:
            results[key] = func(*args, **kwargs)
            return results[key]

    return new_func


def cache_method_result(method):
    method_name = method.__name__

    cache_attr = u"_cache_" + method_name

    @wraps(method)
    def wrapper(self, *args, **kwargs):
        try:
            cache = getattr(self, cache_attr)
        except AttributeError:
            cache = dict()
            setattr(self, cache_attr, cache)

        key = args + tuple([(k, v) for k, v in kwargs.items()])

        if key in cache:
            return cache[key]
        else:
            return cache.setdefault(key, method(self, *args, **kwargs))

    return wrapper


def cache_until_switch_thread_safe(func):
    u"""
    Кеширует результаты выполнения функции, обновляет результаты если сменилась база.
    Все аргументы должны быть hashable. tf значит thread safe
    """
    results = {}
    global_lock = RLock()
    locks = {}

    def get_key_lock(key):
        try:
            return locks[key]
        except KeyError:
            locks[key] = RLock()
            return locks[key]

    def clean_cache(**kwargs):
        try:
            global_lock.acquire()
            try:
                for l in locks.values():
                    l.acquire()
                for k in results.keys():
                    del results[k]
            finally:
                for l in locks.values():
                    l.release()
        finally:
            global_lock.release()
    data_updated.connect(clean_cache, weak=False)

    @wraps(func)
    def new_func(*args, **kwargs):
        try:
            global_lock.acquire()
            key = args + tuple([(k, v) for k, v in kwargs.items()])
            key_lock = get_key_lock(key)
        finally:
            global_lock.release()

        try:
            key_lock.acquire()
            try:
                return results[key]
            except KeyError:
                results[key] = func(*args, **kwargs)
                return results[key]
        finally:
            key_lock.release()

    return new_func


# Писать в удаленных кэш нужно всегда в потоке https://st.yandex-team.ru/EXRASP-11023
def global_cache_methodcall(name, *args, **kwargs):
    def call_alias(alias):
        try:
            cache = caches[alias]
        except:
            log.exception(u'Ошибка при открытии кэша %s в потоке' % alias)
            return

        try:
            method = getattr(cache, name)
            method(*args, **kwargs)
        except:
            log.exception(u'Ошибка при вызове метода %s кэша %s в потоке' % (name, alias))

        cache.close()

    for alias, params in settings.CACHES.items():
        # Здесь мы хотим чтобы значение сохранилось не только в 'default' кэш
        # но и в другие доступные нам кэши, с Memcached бэкендом.
        # Смысл (возможно!) в том, чтобы морды и колдунщики могли обмениваться
        # информацией о тарифах.
        # Так как у нас есть еще и локальные кэши, в которые мы здесь ничего
        # сохранять не хотим - отфильтровываем их.
        if params['BACKEND'] == 'travel.avia.library.python.common.utils.memcache_backend.MemcachedCache':
            start_new_thread(call_alias, (alias,))


def global_cache_add(key, data, timeout=None):
    global_cache_methodcall('add', key, data, timeout)


def global_cache_set(key, data, timeout=None):
    global_cache_methodcall('set', key, data, timeout)


def host_cache_set_key(host, key, value, timeout):
    cache = caches[dc_cache_name(host)]

    try:
        cache.set(key, value, timeout)

    except Exception:
        log.exception('Storing key to cache error')

    finally:
        cache.close()


def cluster_cache_set(key, value, timeout):
    log.debug(u'Store key: %s value: %s', key, value)

    global_cache_set(key, value, timeout)

    if not settings.CLUSTER_CACHE_HOSTS:
        return

    workers = [
        run_in_thread(host_cache_set_key, (host, key, value, timeout))
        for host in settings.CLUSTER_CACHE_HOSTS
    ]

    for worker in workers:
        worker.join()


def cached(key_func, timeout=settings.CACHES['default']['LONG_TIMEOUT'],
           cache_root=settings.CACHEROOT, is_localized=False):
    """
    Кеширующий декоратор.
    is_localized - если значение True, то результат вызова декорируемой функции
                   зависит от текущего языка запроса
    """

    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            key = cache_root + key_func(*args, **kwargs)

            if is_localized:
                key += '/' + get_language()

            if key is None:
                return func(*args, **kwargs)  #: не кешируем запрос

            value = _cache_get(key)

            if value is None:
                value = func(*args, **kwargs)
                _cache_set(key, value, timeout)

            return value

        return wrapper
    return decorator


@retry(
    wait_exponential_multiplier=100,
    wait_exponential_max=500,
    stop_max_attempt_number=3,
)
def _cache_set(key, value, timeout):
    cache.set(key, value, timeout)


@retry(
    wait_exponential_multiplier=100,
    wait_exponential_max=500,
    stop_max_attempt_number=3,
)
def _cache_get(key):
    return cache.get(key)
