# -*- mode: python; coding: utf-8 -*-
import random
import logging

import MySQLdb

from typing import Optional  # noqa

from travel.avia.library.python.common.utils.dcutils import Replica  # noqa

log = logging.getLogger(__name__)


class ReplicaState(object):
    def __init__(self, replica, alive, error=None):
        # type: (Replica, bool, Optional[str]) -> None
        self.replica = replica
        self.alive = alive
        self.error = error

    def __repr__(self):
        return "<%s replica=%r alive=%r error=%r>" % (
            self.__class__.__name__,
            self.replica,
            self.alive,
            self.error,
        )

    @classmethod
    def build_prioritized_list(cls, replicas, state_store, ignore_master=False):
        states = []

        for replica in replicas:
            # исключаем мастер из списка доступных реплик
            if ignore_master and replica.is_master:
                continue

            states.append(
                cls(replica, state_store.is_alive(replica)),
            )

        states.sort(key=cls.prioritized_sort_key)

        return states

    @classmethod
    def prioritized_sort_key(cls, s):
        # Сначала живые, потом в нашем ДЦ, потом по убыванию приоритета, потом случайно
        return (
            0 if s.alive else 1,
            s.replica.priority,
            0 if s.replica.dc_local else 1,
            random.random()
        )


class FailedToConnect(Exception):
    def __init__(self, states):
        self.states = states

    def __str__(self):
        return "Failed to connect with states %r" % self.states


def connect(replicas, state_store, **kwargs):
    prioritized_list = ReplicaState.build_prioritized_list(replicas, state_store)

    log.info('States: %r', prioritized_list)

    for rs in prioritized_list:
        replica = rs.replica
        try:
            conn = MySQLdb.connect(**specified_replica_kwargs(replica, kwargs))

            log.info('Connected to %s', replica)

            return conn

        except MySQLdb.OperationalError, e:
            log.warning('Could not connect to %s: %s', replica, e)

            state_store.set_error(replica, e)

            rs.error = e

    raise FailedToConnect(prioritized_list)


def actualize_replicas_info(replicas, state_store, kwargs):
    log.info('Actualize replica information')
    for replica in replicas:  # type: Replica
        try:
            actualize_replica_is_master(
                replica,
                specified_replica_kwargs(replica, kwargs)
            )
            log.info('%r is master: %s', replica, replica.is_master)
        except MySQLdb.OperationalError, e:
            log.warning('Could not connect to %s: %s', replica, e)
            state_store.set_error(replica, e)


def actualize_replica_is_master(replica, kwargs):
    with MySQLdb.connect(**kwargs) as cursor:
        cursor.execute('show slave status')
        replica.is_master = not cursor.fetchall()


def specified_replica_kwargs(replica, kwargs):
    kwargs = kwargs.copy()
    kwargs['host'] = replica.host
    return kwargs
