# -*- coding: utf-8 -*-

import logging
from datetime import datetime

from django.core.cache import cache
from django.conf import settings

from travel.avia.library.python.common.models.currency import Currency
from travel.avia.library.python.common.models.geo import Settlement
from travel.avia.library.python.common.utils.caching import cluster_cache_set

log = logging.getLogger(__name__)

CURRENCY_RATES_GEOID = Settlement.MOSCOW_GEO_ID
CURRENCY_RATES_CACHE_KEY = 'currency_rates'

NATIONAL_CURRENCY_RATES_GEOID = {
    'ru': Settlement.MOSCOW_GEO_ID,
    'ua': Settlement.KIEV_GEO_ID,
    'tr': Settlement.ISTANBUL_GEO_ID,
    'kz': 163,
    'com': 10376,
}

CURRENCY_RATES_CACHE_TIMEOUT = 60 * 60 * 24


class ConverterCache(object):
    def __init__(self, rates, actual_datetime, national_version):
        self.rates = rates
        self.actual_datetime = actual_datetime
        self.national_version = national_version

    @classmethod
    def load(cls, national_version='ru'):
        return cache.get(cls._get_key(national_version))

    @classmethod
    def build(cls, national_version='ru'):
        currencies = Currency.objects.all()
        src, rates = Currency.fetch_rates(
            currencies,
            cls.get_currency_geoid(national_version),
            base_currency=cls.get_national_currency(national_version)
        )

        if src is None:
            raise RuntimeError('no data from currency converter')

        return cls(rates, datetime.now(), national_version)

    def save(self, cluster=False):
        cache_set = cluster_cache_set if cluster else cache.set

        cache_set(
            key=self._get_key(self.national_version),
            value=self,
            timeout=CURRENCY_RATES_CACHE_TIMEOUT
        )

    @classmethod
    def _get_key(cls, national_version):
        return '/yandex/rasp/%s/%s' % (
            CURRENCY_RATES_CACHE_KEY, national_version
        )

    @classmethod
    def get_currency_geoid(cls, national_version):
        return NATIONAL_CURRENCY_RATES_GEOID[national_version]

    @classmethod
    def get_national_currency(cls, national_version):
        return settings.NATIONAL_CURRENCIES[national_version]
