# -*- coding: utf-8 -*-
from __future__ import absolute_import

"""Набор функция для работы с datetime"""

import logging
import time
import calendar
import re
import warnings
from collections import namedtuple
from datetime import date, datetime, timedelta
from itertools import dropwhile, groupby, takewhile
from time import mktime

import pytz
from pytils.dt import ru_strftime

from django.utils import translation

from travel.avia.library.python.common.utils.warnings import RaspDeprecationWarning
from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.utils.text import NBSP
from travel.avia.library.python.common.xgettext.common import xgettext_weekday, xgettext_weekday_short, xgettext_month, \
    xgettext_month_genitive, xgettext_month_short
from travel.avia.library.python.common.xgettext.i18n import tngettext

log = logging.getLogger(__name__)


MSK_TZ = pytz.timezone('Europe/Moscow')
KIEV_TZ = pytz.timezone('Europe/Kiev')
UTC_TZ = pytz.timezone('UTC')

DAYS_TEXT_LIMIT_MIN = 7
DAYS_TEXT_LIMIT_MAX = 20
DAYS_TO_PAST = 30


# заполняем числами - сколько дней в каждом месяце этого года
days_in_months = []


def get_days_in_months(fresh=False):
    global days_in_months
    if days_in_months and not fresh:
        return days_in_months

    # Чтобы в високосные года брался более актуальный февраль
    year = (datetime.today() + timedelta(30*8)).year

    days_in_months = []
    for i in range(1, 12):
        days_in_months.append((datetime(year, i+1, 1) -
                               timedelta(1)).day)
    days_in_months.append(31)
    return days_in_months


def get_short_yeardays(yeardays, mask=get_days_in_months(True)):
    """ Возвращает год в виде строки.
        В каждом месяце столько дней сколько должно быть, а не 31.
    """
    res = ''
    for i in range(len(mask)):
        days = get_days_in_months()[i]
        res += yeardays[i*31:i*31 + days]
    return res


def get_full_yeardays(yeardays):
    """ Возвращает год в виде строки.
        В каждом месяце ровно 31 день.
    """
    res = ''
    for i in range(len(get_days_in_months())):
        days = get_days_in_months()[i]
        res += yeardays[:days] + '0'*(31-days)
        yeardays = yeardays[days:]
    return res


def get_moved_year_days(year_days, sds):
    """ Возвращает сдвинутые на sds дней year_days
    """
    return year_days[-sds:] + year_days[:-sds]


def get_months(year_days, sds=0):
    """ Возвращает числа из строки year_days уже сдвинутые на sds дней
    """
    year_days = get_short_yeardays(year_days)
    new_year_days = year_days[-sds:] + year_days[:-sds]  # new means shifted

    months = []
    for month in range(1, 13):
        prev_days_sum = sum(get_days_in_months()[:month-1])
        val = new_year_days[prev_days_sum:
                            prev_days_sum + get_days_in_months()[month-1]]
        new_val = ''
        for i in range(len(val)):
            new_val += val[len(val)-i-1]
        months.append(int(new_val, 2))
    return months


def make_calendar(daily_data_by_month):
    """
    Возвращает календарь (год, разбитый на месяцы и недели)
    с данными, подключенными к дням, в виде списка месяцев.
    Месяц - это список недель. Каждая неделя - список пар
    (day, day_data), где day и day_data равны None, если
    эти дни за пределами месяца.

    Год подразумевается текущим, если передан как None
    daily_data_by_month - список списков, по списку на месяц.
    """

    months = []

    for first_month_day, month_data in daily_data_by_month:
        month = first_month_day.month
        weeks = calendar.monthcalendar(first_month_day.year, month)

        for week in weeks:
            # i нужен, чтобы модифицировать список i на месте (экономим память)
            for i, day in enumerate(week):
                # Ноль означает, что день за пределами месяца
                if day:
                    # Пробуем получить данные
                    attrs = {
                        'holiday': i > 4,
                    }

                    if month_data is not None:
                        day_data = month_data[day - 1]
                    else:
                        day_data = None

                    week[i] = (date(first_month_day.year, month, day), day_data, attrs)

                else:
                    week[i] = (None, None, None)

        months.append({
            'name': uni_strftime(u'%B', first_month_day),
            'weeks': weeks,
            'first_day': first_month_day,
            'data': month_data,
        })

    return months


def get_first_month_days(months=12, today=None):
    """ Формирует список дат - первых дней месяцев, начиная от предыдущего перед текущим
        Вход: количество месяцев
        Выход: список первых дней месяцев """
    if not today:
        today = environment.today()
    start = date(today.month == 1 and today.year-1 or today.year,
                 today.month == 1 and 12 or today.month-1,
                 1)

    return (
        date(
            start.month + m > 12 and start.year + 1 or start.year,
            start.month + m == 12 and 12 or (start.month + m) % 12,
            1
        ) for m in xrange(0, months)
    )


def fromSQLdatetime(datetimeStr):
    """Возвращает дату-время из SQL-формата
    Вход: дата-время в формате yyyy-mm-dd hh:mm:ss
    Выход: объект datetime"""
    return datetime(*time.strptime(datetimeStr, "%Y-%m-%d %H:%M:%S")[:6])


def fromSQLDate(dateStr):
    """Возвращает дату из SQL-формата
    Вход: дата в формате yyyy-mm-dd
    Выход: объект date"""
    return datetime(*time.strptime(dateStr, "%Y-%m-%d")[:3])


def parse_date(dt):
    if dt is None:
        return None

    return date(*(int(e) for e in dt.split('-')))


def clean_datetime(datetime_str):
    """ Преобразует дату время из SQL формата в datetime
        Вход: дата-время в формате yyyy-mm-dd hh:mm:ss
        Выход: объект datetime или None если входное значение невалидно
    """
    try:
        return fromSQLdatetime(datetime_str)
    except (ValueError, TypeError):
        return None


def get_near_date_future(year_days, today=None):
    warnings.warn('Use RunMask.first_run # 2015-04-14', RaspDeprecationWarning, stacklevel=2)
    today = today or environment.today()
    return RunMask.first_run(year_days, today)


def timedelta2minutes(delta):
    return delta.total_seconds() / 60


def timedelta2hours(delta):
    return delta.total_seconds() / 3600


def timedelta_to_str_hours_and_minutes(td):
    """
    :param td: datetime.timdedelta
    :return: unicode: HH:MM

    Ex: timedelta(hours=1, minutes=42) => '01:42'
    """
    hours, min_sec = divmod(td.total_seconds(), 3600)
    minutes, _ = divmod(min_sec, 60)

    return u'{:02d}:{:02d}'.format(int(hours), int(minutes))


MSK_TIMEZONE = 'Europe/Moscow'


def get_pytz(time_zone):
    try:
        return pytz.timezone(str(time_zone))
    except pytz.UnknownTimeZoneError:
        return MSK_TZ


def get_timezone_from_point(point):
    time_zone = None

    if isinstance(point, dict):
        time_zone = point['time_zone'] or MSK_TIMEZONE
    elif isinstance(point, basestring):
        return point
    else:
        if hasattr(point, 'settlement') and point.settlement:
            time_zone = point.settlement.time_zone

        if time_zone is None:
            time_zone = point.time_zone

        if time_zone is None and hasattr(point, 'region'):
            time_zone = point.region and point.region.time_zone

    if time_zone is None:
        time_zone = MSK_TIMEZONE

    return time_zone


def get_msk_time_zone(point, msk_datetime=None, local_datetime=None,
                      utc_datetime=None):
    """ Возвращает смещение географического объекта в минутах относительно москвы
    с учетом перехода его на зимнее время,
    принимется либо местное время либо Московское

    Объект должен быть словарем содержищим ключи 'time_zone' и 'switch_time',
    или объектом содержищим такие аттрибуты,
    или объектом django моделей Settlement, Station или Region

    Если не указано время, то берем текущее.

    """

    if msk_datetime is None and local_datetime is None:
        msk_datetime = datetime.now()

    time_zone = get_timezone_from_point(point)

    loc_tz = get_pytz(time_zone)

    utc_tz = pytz.utc

    if msk_datetime is not None:
        msk_datetime = MSK_TZ.localize(msk_datetime)
        local_datetime = msk_datetime.astimezone(loc_tz)
    elif local_datetime is not None:
        local_datetime = loc_tz.localize(local_datetime)
        msk_datetime = local_datetime.astimezone(MSK_TZ)
    elif utc_datetime is not None:
        utc_datetime = utc_tz.localize(utc_datetime)
        msk_datetime = utc_datetime.astimezone(MSK_TZ)
        local_datetime = utc_datetime.astimezone(loc_tz)

    return timedelta2minutes(local_datetime.replace(tzinfo=None) - msk_datetime.replace(tzinfo=None))


def get_local_time(point, msk_datetime=None, utc_datetime=None):
    """ Конвертирует Московское время в местное c учетом того,
    переходит географический объект на зимнее время или нет.

    Объект должен быть словарем содержищим ключи 'time_zone' и 'switch_time',
    или объектом содержищим такие аттрибуты,
    или объектом django моделей Settlement, Station или Region
    """

    time_zone = get_timezone_from_point(point)

    if msk_datetime:
        loc_tz = get_pytz(time_zone)
        msk_datetime = MSK_TZ.localize(msk_datetime)
        local_datetime = msk_datetime.astimezone(loc_tz)
    elif utc_datetime:
        utc_tz = pytz.utc
        loc_tz = get_pytz(time_zone)
        utc_datetime = utc_tz.localize(utc_datetime)
        local_datetime = utc_datetime.astimezone(loc_tz)
    else:
        raise ValueError("No datetime")

    return local_datetime


class Localizer(object):
    def __init__(self, point):
        self.loc_tz = get_pytz(get_timezone_from_point(point))

    def localize(self, msk_datetime):
        if self.loc_tz == MSK_TZ:
            return MSK_TZ.localize(msk_datetime)
        else:
            return MSK_TZ.localize(msk_datetime).astimezone(self.loc_tz)


def get_msk_time(point, local_datetime):
    """ Конвертирует местное время в Московкое c учетом того,
    переходит географический объект на зимнее время или нет.

    Объект должен быть словарем содержищим ключи 'time_zone' и 'switch_time',
    или объектом содержищим такие аттрибуты,
    или объектом django моделей Settlement, Station или Region
    """

    time_zone = get_timezone_from_point(point)

    loc_tz = get_pytz(time_zone)

    local_datetime = loc_tz.localize(local_datetime)
    return local_datetime.astimezone(MSK_TZ)


def get_utc_time(point, local_datetime):
    if isinstance(point, dict):
        time_zone = point['time_zone'] or MSK_TIMEZONE
    else:
        time_zone = point.time_zone or MSK_TIMEZONE
        # На то случай если это None

    loc_tz = get_pytz(time_zone)
    utc_tz = pytz.utc

    local_datetime = loc_tz.localize(local_datetime)
    return local_datetime.astimezone(utc_tz)


def daterange(begin, end, delta=timedelta(1), include_end=False):
    """Form a range of dates and iterate over them.

    Arguments:
    begin -- a date (or datetime) object; the beginning of the range.
    end   -- a date (or datetime) object; the end of the range.
    delta -- (optional) a timedelta object; how much to step each iteration.
             Default step is 1 day.

    Usage:

    """
    if not isinstance(delta, timedelta):
        delta = timedelta(delta)

    ZERO = timedelta(0)

    if begin < end:
        if delta <= ZERO:
            raise StopIteration
        if include_end:
            test = end.__ge__
        else:
            test = end.__gt__
    else:
        if delta >= ZERO:
            raise StopIteration
        if include_end:
            test = end.__le__
        else:
            test = end.__lt__

    while test(begin):
        yield begin
        begin += delta


def fix_date(d, today):
    """Выставляет дате год с учетом наших ограничений"""
    if d and not((d.month, d.day) == (2, 29)):
        left_border = today - timedelta(DAYS_TO_PAST)

        d = d.replace(year=today.year)

        if d < left_border:
            d = d.replace(year=today.year + 1)

    return d


class DaysText(namedtuple('DaysText', 'raw has_more')):
    def __unicode__(self):
        if self.has_more:
            return self.raw + u', …'

        return self.raw


class RunMask(object):
    """ Маска дней хождения"""

    MASK_LENGTH = 12 * 31
    EMPTY_YEAR_DAYS = '0' * MASK_LENGTH
    ALL_YEAR_DAYS = '1' * MASK_LENGTH

    cached = {}  # Маски дат для разных дней
    extrapolation_forbidden = False

    def __init__(self, mask=None, today=None, days=None, strict=False):
        if hasattr(mask, '__int__'):
            self.mask = int(mask)
        elif mask is None or mask == '':
            self.mask = 0
        else:
            if len(mask) == self.MASK_LENGTH:
                try:
                    self.mask = int(mask, 2)
                except (TypeError, ValueError):
                    if strict:
                        raise ValueError("Str mask must have %s chars instead of %s" % (self.MASK_LENGTH, len(mask)))
                    else:
                        log.error("Bad runmask %r" % mask)
                        self.mask = 0
            else:
                if strict:
                    raise ValueError("Str mask must have %s chars instead of %s" % (self.MASK_LENGTH, len(mask)))
                else:
                    log.error("Bad runmask %r" % mask)
                    self.mask = 0

        if days:
            for day in days:
                self[day] = 1

        self.set_today(today)

    def get_today(self):
        return self._today

    def set_today(self, today):
        if isinstance(today, datetime):
            today = today.date()

        self._today = today

        # для разных переборов
        if self._today:

            if self._today in self.cached:
                self.all_days, self.date_masks = self.cached[self._today]
            else:
                start_day = self._today - timedelta(DAYS_TO_PAST)

                try:
                    end_day = start_day.replace(year=start_day.year + 1)
                except ValueError:
                    end_day = (start_day + timedelta(1)).replace(year=start_day.year + 1)

                self.all_days = list(daterange(start_day, end_day))

                self.date_masks = [(d, self.date_mask(d)) for d in self.all_days]
                self.cached[self._today] = self.all_days, self.date_masks
        else:
            self.date_masks = None

    today = property(get_today, set_today, doc=u"""
    Устанавливает или снимает привязку к дате.
    Нужен для генереции дат хождений.
    """)

    def __str__(self):
        return "".join(self.mask & (1 << (self.MASK_LENGTH - 1 - d)) and '1' or '0'
                       for d in range(self.MASK_LENGTH))

    def __int__(self):
        return self.mask

    def __nonzero__(self):
        return self.mask != 0

    @classmethod
    def index(cls, date_):
        return (date_.month - 1) * 31 + (date_.day - 1)

    @classmethod
    def date_mask(cls, date_):
        # Год у нас начится с MSB, так как получаем маску мы с помщью int()
        return 1 << (cls.MASK_LENGTH - 1 - cls.index(date_))

    def __getitem__(self, date_):
        return bool(self.mask & self.date_mask(date_))

    def __setitem__(self, date_, runs):
        mask = self.date_mask(date_)

        if runs:
            self.mask |= mask
        else:
            self.mask = (self.mask | mask) ^ mask

    def __contains__(self, date_):
        return self[date_]

    def issubset(self, other):
        return self.mask & other.mask == self.mask

    def issuperset(self, other):
        return self.mask & other.mask == other.mask

    def difference(self, other):
        u"""Разность множеств дней хождений"""
        return self - other

    def __sub__(self, other):
        return (self | other) ^ other

    def __hash__(self):
        return hash((hash(self.today), self.mask))

    def __or__(self, other):
        return RunMask(self.mask | other.mask, today=self._today)

    def __and__(self, other):
        return RunMask(self.mask & other.mask, today=self._today)

    def __eq__(self, other):
        return self.mask == other.mask and self.today == other.today

    def __ne__(self, other):
        return not self.__eq__(other)

    def __xor__(self, other):
        return RunMask(self.mask ^ other.mask, today=self._today)

    def __repr__(self):
        if not self.today:
            self.today = date.today()
        dates = self.dates()
        days = u", ".join(d.strftime("%Y-%m-%d") for d in dates[:10])
        if len(dates) > 10:
            days += u"..."

        return u"RunMask[%s]" % days

    @classmethod
    def range(cls, start, end, today=None, include_end=False):
        mask = 0

        for d in daterange(start, end, include_end=include_end):
            mask |= cls.date_mask(d)

        return RunMask(mask, today=today)

    def portion(self, start_date, length):
        portion_mask = self.range(start_date, start_date + timedelta(length))

        return self & portion_mask

    def dates(self, past=True):
        u"""
        Список дат по которым ходит маршрут
        @param past: Брать ли дни раньше, чем self.today.
        """

        if self._today is None:
            raise ValueError("Method dates is not available on RunMask's without today")

        if past:
            return [d for d, d_mask in self.date_masks if self.mask & d_mask]
        else:
            return [d for d, d_mask in self.date_masks if self.mask & d_mask and d >= self._today]

    def iter_dates(self, past=True):
        u"""
        Список дат по которым ходит маршрут
        @param past: Брать ли дни раньше, чем self.today.
        """

        if self._today is None:
            raise ValueError("Method dates is not available on RunMask's without today")

        if past:
            return (d for d, d_mask in self.date_masks if self.mask & d_mask)
        else:
            return (d for d, d_mask in self.date_masks if self.mask & d_mask and d >= self._today)

    def shifted(self, shift):
        u"""
        Сдвигает маску на shift дней вперед
        """
        if shift:
            m = RunMask(today=self.today)

            for d in self.dates():
                m[d + timedelta(shift)] = 1
        else:
            m = RunMask(self, today=self.today)

        m.extrapolation_forbidden = self.extrapolation_forbidden

        return m

    _ranges = {}

    @classmethod
    def range_equal(cls, mask1, mask2, period_start, period_end):
        if (period_start, period_end) in cls._ranges:
            compare_range = cls._ranges[(period_start, period_end)]
        else:
            compare_range = RunMask.range(period_start, period_end)
            cls._ranges[(period_start, period_end)] = compare_range

        mask1 = mask1 & compare_range
        mask2 = mask2 & compare_range
        return mask1 == mask2

    @classmethod
    def mask_day_index(cls, dt):
        return (dt.day - 1) + (dt.month - 1) * 31

    @classmethod
    def runs_at(cls, year_days, dt):
        if not year_days:
            return False

        if len(year_days) != cls.MASK_LENGTH:
            log.error('Bad runmask %r' % year_days)
            return False

        return year_days[cls.mask_day_index(dt)] == '1'

    @classmethod
    def first_run(cls, year_days, today):
        u"""Вычисляет первый или последний день хождения"""

        year_days = str(year_days)

        if len(year_days) != cls.MASK_LENGTH:
            log.error('Bad runmask %r' % year_days)
            return None

        def safe_date(year, month, day):
            try:
                return date(year, month, day)
            except ValueError:
                assert (month == 2 and day == 29)

                return date(year, 3, 1)

        # Ищем первую дату хождения после сегодняшнего дня
        def first():
            day_position = (today.month - 1) * 31 + (today.day - 1)

            year = today.year

            try:
                first_day = year_days.index('1', day_position)
            except ValueError:
                year += 1

                try:
                    first_day = year_days.index('1')
                except ValueError:
                    return None

            month0, day0 = divmod(first_day, 31)

            return safe_date(year, month0 + 1, day0 + 1)

        def last():
            day_position = (today.month - 1) * 31 + (today.day - 1)

            year = today.year

            try:
                last_day = year_days.rindex('1', 0, day_position)
            except ValueError:
                year -= 1

                try:
                    last_day = year_days.rindex('1')
                except ValueError:
                    return None

            month0, day0 = divmod(last_day, 31)

            return safe_date(year, month0 + 1, day0 + 1)

        first_run = first()

        if first_run is None:
            return None

        if first_run - today > timedelta(365 - DAYS_TO_PAST):
            first_run = last()

        return first_run

    def format_days_text(self, days_limit=True, shift=0, lang=None):
        lang = lang or translation.get_language()

        all_dates = self.dates()
        if not all_dates:
            return None

        if shift:
            all_dates = [d + timedelta(days=shift) for d in all_dates]

        if days_limit is None:
            # Не ограничиваем по датам и количеству
            return DaysText(
                group_days([(d.day, d.month) for d in all_dates], lang),
                False
            )

        dates_limit = self.today + timedelta(days=300)
        limited_dates = list(
            takewhile(lambda d: d < dates_limit, all_dates)
        )

        if not limited_dates:
            limited_dates = all_dates

        if days_limit:
            # Даты хождения не в прошлом
            dates = list(
                dropwhile(lambda d: d < self.today, limited_dates)
            )

            # Если их меньше минимума, то берем просто минимум последних
            if len(dates) < DAYS_TEXT_LIMIT_MIN:
                dates = limited_dates[-DAYS_TEXT_LIMIT_MIN:]
        else:
            dates = limited_dates

        return DaysText(
            group_days([(d.day, d.month) for d in dates[:DAYS_TEXT_LIMIT_MAX]], lang),
            len(dates) > DAYS_TEXT_LIMIT_MAX
        )


class StrictRunMask(RunMask):
    def __init__(self, mask=None, today=None, days=None):
        RunMask.__init__(self, mask, today, days, strict=True)


def get_schedule_change_period(today):
    warnings.warn('Use this TrainSchedulePlan.get_schedule_end_period ## 2015-04-14',
                  RaspDeprecationWarning, stacklevel=2)

    # RASP-4269

    # Последнее воскресенье мая, но не ранее, чем сегодня
    may_end = today.replace(month=5, day=31)

    last_sun = may_end - timedelta((may_end.weekday() + 1) % 7)

    # Берем следующее последнее воскресенье мая
    if last_sun < today:
        may_end = may_end.replace(year=today.year + 1)
        last_sun = may_end - timedelta((may_end.weekday() + 1) % 7)

    april_first = last_sun.replace(month=4, day=1)

    return april_first, last_sun


def group_days(days, lang='ru'):
    if not days:
        return u''

    if not isinstance(days[0], tuple):
        days = [tuple(int(p) for p in d.split(".")) for d in days.split(",")]

    grouped = groupby(days, key=lambda d: d[1])

    daysparts = []

    for month, days in grouped:
        days = ", ".join(str(d[0]) for d in days)
        daysparts.append(u"%s %s" % (days, xgettext_month_short(month, lang)))

    return ", ".join(daysparts)


def to_msk(dt):
    return dt.astimezone(MSK_TZ).replace(tzinfo=None)


class FuzzyDateTime(object):
    STEP = 5
    HALF_STEP = float(STEP) / 2

    timetuple = True

    def __init__(self, dt, round_dt=None):
        self.dt = dt
        self.round_dt = round_dt or dt

    @classmethod
    def fuzz(cls, dt):
        # Разница с округленным значением
        round_diff = (dt.minute + cls.HALF_STEP) % cls.STEP - cls.HALF_STEP

        round_dt = dt - timedelta(
            microseconds=dt.microsecond,
            seconds=dt.second,
            minutes=round_diff
        )

        return cls(dt, round_dt)

    def __cmp__(self, other):
        if isinstance(other, self.__class__):
            return cmp(self.round_dt, other.round_dt)

        if isinstance(other, datetime):
            return cmp(self.round_dt, other)

        return NotImplemented

    def astimezone(self, *args, **kwargs):
        return self.__class__(self.round_dt.astimezone(*args, **kwargs))

    def replace(self, *args, **kwargs):
        return self.__class__(self.round_dt.replace(*args, **kwargs))

    def __add__(self, other):
        return self.round_dt + other

    def __sub__(self, other):
        if isinstance(other, self.__class__):
            return self.round_dt - other.round_dt

        return self.round_dt - other

    def __rsub__(self, other):
        if isinstance(other, self.__class__):
            return other.round_dt - self.round_dt

        return other - self.round_dt

    def __str__(self):
        return self.strftime('%Y-%m-%d')

    def __getattr__(self, item):
        return getattr(self.round_dt, item)


def astimezone(dt, city):
    # tz передается как город

    if not city:
        # местное время
        return dt

    return dt.astimezone(get_pytz(city.time_zone))


def _uni_strftime(fmt, dt, inflected=False, lang=None):
    fmt = re.sub(u'(?<![\.\d\-])%d(?![\.\d\-])', unicode(dt.day), fmt, flags=re.M + re.U)
    if u'%b' in fmt or u'%B' in fmt:
        fmt = fmt.replace(u'%d', unicode(dt.day))

    weekday = dt.weekday() + 1  # пн = 1,..., вс = 7

    lang = lang or translation.get_language()

    month_func = xgettext_month_genitive if inflected else xgettext_month

    fmt = fmt.replace(u'%a', xgettext_weekday_short(weekday, lang))
    fmt = fmt.replace(u'%A', xgettext_weekday(weekday, lang))
    fmt = fmt.replace(u'%b', xgettext_month_short(dt.month, lang))

    if '%B' in fmt:
        # Для этих языков по умолчанию писать месяца с маленькой буквы
        if lang in ('ru', 'uk'):
            month = month_func(dt.month, lang).lower()
        else:
            month = month_func(dt.month, lang)

        fmt = fmt.replace(u'%B', month)

    return dt.strftime(fmt.encode('utf-8')).decode('utf-8')


def uni_strftime(fmt, dt, lang=None):
    lang = lang or translation.get_language()

    fmt = unicode(fmt)

    if isinstance(dt, FuzzyDateTime):
        dt = dt.round_dt

    inflected = u'%d %B' in fmt or (u'%d' + NBSP + u'%B') in fmt

    if lang == 'ru':
        return ru_strftime(fmt, dt, inflected=inflected)

    else:
        return _uni_strftime(fmt, dt, inflected=inflected, lang=lang)


class DateTimeFormatter(object):
    def __init__(self, dt, lang=None):
        self.dt = dt
        self.lang = lang

    def L(self, format=u'%d %B'):
        return uni_strftime(format, self.dt, self.lang)

    def __format__(self, format_):
        return self.L(format_)


def human_date_without_year(dt):
    return uni_strftime(u'%d %B', dt)


def human_date(dt):
    if date.today().year == dt.year:
        return human_date_without_year(dt)

    return uni_strftime(u'%d %B %Y', dt)


def human_date_with_time(dt):
    return uni_strftime(u'%d %B %H:%M', dt)


def human_duration(duration):
    if not duration or duration < timedelta():
        return None

    minutes = (duration.seconds + 1) / 60
    hours = 0
    days = duration.days

    if minutes > 59:
        hours = minutes / 60
        minutes = minutes % 60

    # Если есть дни, минуты округляем до часов
    if days:
        hours += (minutes + 60 - 1) / 60
        minutes = 0

        # Если получилось 24 часа, то прибавляем день и убираем часы
        if hours == 24:
            days += 1
            hours = 0

    blocks = []

    if days:
        # Продолжительность действия: 1 день. Может являться частью фразы, например, 1 день 2 часа
        blocks.append(tngettext(days, u'<days /> день', u'<days /> дня', u'<days /> дней', **{'days': str(days)}))

    if hours:
        # Продолжительность действия: 1 час. Может являться частью фразы, например, 1 день 2 часа
        blocks.append(tngettext(hours,  u'<hours /> ч',  u'<hours /> ч',  u'<hours /> ч', **{'hours': str(hours)}))

    if minutes and len(blocks) < 2:
        # Продолжительность действия: 1 минута. Может являться частью фразы, например, 2 часа 3 минуты
        blocks.append(tngettext(minutes, u'<minutes /> мин', u'<minutes /> мин', u'<minutes /> мин', **{'minutes': str(minutes)}))

    return u' '.join(b.replace(' ', NBSP) for b in blocks)


def get_time_zone_choices():
    # Немного американской тематики
    summer_day = datetime(date.today().year, 7, 4)
    # Новый год, новый год
    winter_day = datetime(date.today().year, 1, 1)

    if datetime.today() > summer_day:
        winter_day = datetime(date.today().year, 12, 1)

    msk_datetime_s = MSK_TZ.localize(summer_day)
    msk_datetime_w = MSK_TZ.localize(winter_day)

    def get_diff_hours(tz):
        loc_tz = pytz.timezone(tz)
        loc_datetime = msk_datetime_s.astimezone(loc_tz)
        diff = loc_datetime.replace(tzinfo=None) - msk_datetime_s.replace(tzinfo=None)
        summer_hours = diff.days * 24 + diff.seconds / 3600.0

        loc_datetime = msk_datetime_w.astimezone(loc_tz)
        diff = loc_datetime.replace(tzinfo=None) - msk_datetime_w.replace(tzinfo=None)
        winter_hours = diff.days * 24 + diff.seconds / 3600.0

        return u" %+.2f от Москвы летом, %+.2f от Москвы зимой" % \
            (summer_hours, winter_hours)

    choices = [(unicode(tz), unicode(tz) + get_diff_hours(tz))
               for tz in sorted(pytz.all_timezones)]
    return choices


def to_unix_timestamp(dt):
    """
    Преобразует дату со временем в unix timestamp
    """

    return int(mktime(dt.timetuple()))


def smart_localize(dt, tz):
    """
    Если время может быть как зимним так и летним, берем наибольшее по UTC
    """
    try:
        return tz.localize(dt, is_dst=None)
    except pytz.InvalidTimeError:
        pass

    return max(
        tz.normalize(tz.localize(dt, is_dst=False)),
        tz.normalize(tz.localize(dt, is_dst=True))
    )
