# -*- coding: utf-8

u"""
Тут нельзя использовать джангу, т.к. эти функции используются в настройках(local_settings.py),
которые вызываются еще до инициализации джанги.
"""
from __future__ import print_function

import codecs
import json
import os.path
import socket
import sys
import urllib2
from datetime import datetime
from traceback import format_exc
from urllib import urlencode

from travel.avia.library.python.common.utils.http import urlopen

CONDUCTOR_API = 'http://c.yandex-team.ru/api/'

GROUP_ALIASES = {
    'rasp_prestable_front': 'rasp_front',
    'rasp_stable_front': 'rasp_front',

    # FIXME: убрать после переименования групп
    'rasp_front-prestable': 'rasp_front',
    'rasp_front-stable': 'rasp_front'
}


class Conductor(object):
    utf8_stdout = codecs.getwriter('utf8')(sys.stdout)

    def __init__(self, log_file_path, cache_path):
        self.log_file_path = log_file_path
        self.cache_path = cache_path

        self._dc = None

    def log_error(self, *objects):
        objects = tuple([u"%s:" % datetime.today().strftime('%Y-%m-%d %H:%M:%S')] + list(objects))

        try:
            file = codecs.open(self.log_file_path, 'a', encoding='utf8')
            print(*objects, file=file)
            file.close()
        except IOError:
            print(*objects, file=self.utf8_stdout)

            self.utf8_stdout.flush()

    def _get_data(self, request):
        url = CONDUCTOR_API + request

        data = urlopen(url, timeout=5).read()

        return data

    def _write_cache(self, data, *path):
        full = os.path.join(self.cache_path, *path)

        d = os.path.dirname(full)

        if not os.path.exists(d):
            os.makedirs(d)

        f = open(full, 'w')
        f.write(data)
        f.close()

    def _read_cache(self, *path):
        full = os.path.join(self.cache_path, *path)

        f = open(full)
        data = f.read()
        f.close()

        return data

    def _get_group_hosts(self, group):
        lines = None
        try:
            lines = self._get_data('groups2hosts/' + group + '?fields=fqdn,root_datacenter_name')

            self._write_cache(lines, 'groups2hosts', group)
        except (urllib2.HTTPError, socket.timeout) as e:
            self.log_error(u"Ошибка при скачивании группы хостов %r: %s" % (group, unicode(e)))
        except Exception:
            self.log_error(u"Ошибка при скачивании группы хостов %r: %s" % (group, format_exc()))

        if lines is None:
            self.log_error(u"Берем информацию о группе хостов %r из кеша" % group)
            lines = self._read_cache('groups2hosts', group)

        for line in lines.split('\n'):
            if line:
                host, dc = line.split('\t')

                yield host, dc

    def get_group_hosts(self, group):
        return [host for host, dc in self._get_group_hosts(group)]

    def get_group_hosts_by_dc(self, group):
        hosts = {}

        for host, dc in self._get_group_hosts(group):
            hosts.setdefault(dc, []).append(host)

        return hosts

    def get_group_hosts_by_dc_with_group(self, group):
        hosts = {}

        for host, dc in self._get_group_hosts(group):
            hosts.setdefault(group + "@" + dc, []).append(host)

        return hosts

    def read_hosts(self, hosts):
        info = None

        try:
            data = self._get_data('hosts/%s/?format=json' % ','.join(hosts))

            info = json.loads(data)

            for host_info in info:
                self._write_cache(json.dumps(host_info), 'hosts', host_info['fqdn'])

        except (urllib2.HTTPError, socket.timeout) as e:
            self.log_error(u"Ошибка при скачивании инфомации хостов %s: %s" %
                           (u", ".join(hosts), unicode(e)))
        except Exception:
            self.log_error(u"Ошибка при скачивании инфомации хостов %s: %s" %
                           (u", ".join(hosts), format_exc()))

        if info is None:
            self.log_error(u"Берем информацию о хостах %s из кеша" % (u", ".join(hosts)))
            info = []

            for host in hosts:
                host_data = self._read_cache('hosts', host)
                info.append(json.loads(host_data))

        return info

    def get_hosts_info(self, hosts):
        info = self.read_hosts(hosts)

        return dict(
            (host_info['fqdn'], host_info)
            for host_info in info
        )

    def get_current_dc(self):
        if self._dc is None:
            try:
                hostname = socket.gethostname()

                info = self.get_hosts_info([hostname])

                self._dc = info[hostname]['root_datacenter']
            except:
                self._dc = 'UNKNOWN'

        return self._dc

    def _get_host_info(self, hostname):
        hostinfo = self.get_hosts_info([hostname])[hostname]
        group_alias = GROUP_ALIASES.get(hostinfo['group'])

        if group_alias is not None:
            hostinfo['group'] = group_alias

        return hostinfo

    def get_current_group(self):
        info = self._get_host_info(socket.gethostname())

        return info['group']

    def get_current_dc_with_group(self):
        info = self._get_host_info(socket.gethostname())

        return '{group}@{root_datacenter}'.format(**info)

    def read_hosts_projects(self, hosts):
        info = None

        try:
            data = self._get_data('hosts2projects/%s/?format=json' % ','.join(hosts))

            info = json.loads(data)

            for host, host_info in zip(hosts, info):
                self._write_cache(json.dumps(host_info), 'hosts2projects', host)

        except (urllib2.HTTPError, socket.timeout) as e:
            self.log_error(u"Ошибка при скачивании инфомации хостов %s: %s" %
                           (u", ".join(hosts), unicode(e)))

        except Exception:
            self.log_error(u"Ошибка при скачивании инфомации хостов %s: %s" %
                           (u", ".join(hosts), format_exc()))

        if info is None:
            self.log_error(u"Берем информацию о хостах %s из кеша" % (u", ".join(hosts)))

            info = []

            for host in hosts:
                host_data = self._read_cache('hosts2projects', host)

                info.append(json.loads(host_data))

        return info

    def get_current_project(self):
        hostname = socket.gethostname()

        info = self.read_hosts_projects([hostname])[0]

        return info['name']

    def _get_online_json_data(self, request, path, key):
        try:
            data = self._get_data(request)

        except (urllib2.HTTPError, socket.timeout) as e:
            self.log_error(u"Ошибка при запросе к кондуктору %s: %s" % (request, unicode(e)))

            raise

        try:
            parsed = json.loads(data)

        except Exception as e:
            self.log_error(u"Ошибка при разборе ответа кондуктора %s: %s" % (request, unicode(e)))

            raise

        try:
            self._write_cache(data, path, key)

        except Exception as e:
            self.log_error(
                u"Ошибка при записи в кэш ответа кондуктора %s %s: %s" %
                (path, key, unicode(e))
            )

            raise

        return parsed

    def _get_cached_json_data(self, request, path, key):
        try:
            data = self._read_cache(path, key)

        except Exception as e:
            self.log_error(u"Ошибка при чтении ответа кондутора из кэша %s: %s" %
                           (request, unicode(e)))

            raise

        try:
            parsed = json.loads(data)

        except Exception as e:
            self.log_error(u"Ошибка при разборе ответа кондуктора из кэша %s: %s" %
                           (request, unicode(e)))

            raise

        return parsed

    def _get_json_data(self, path, params, key):
        request = path

        if params:
            request += '?' + urlencode(params)

        online_error = None

        try:
            return self._get_online_json_data(request, path, key)

        except Exception as online_error:
            pass

        cache_error = None

        try:
            return self._get_cached_json_data(request, path, key)

        except Exception as cache_error:
            pass

        raise Exception(u"Ошибка при запросе к кондуктору %s:\nonline: %s\ncache: %s" % (
            request,
            online_error,
            cache_error,
        ))

    def get_mysql_replicainfo(self, name, root_dc=None):
        hosts = self._get_json_data(
            'generator/mysql_d_groupinfo', {'name': name}, name
        )['hosts']

        hosts_info = self.read_hosts([host['name'] for host in hosts])

        if root_dc is None:
            root_dc = self.get_current_dc()

        master_host = None

        replicas = []

        for host, host_info in zip(hosts, hosts_info):
            replica = Replica(
                host['name'], host['dc'], host_info['root_datacenter']
            )

            replica.dc_local = replica.root_dc == root_dc

            if host['class'] == 'master':
                replica.priority = 1
                replica.is_master = True

                master_host = replica.host

            replicas.append(replica)

        return ReplicaInfo(master_host, replicas)

    def get_groups_mysql_replicainfo(self, group, master_group, root_dc=None):
        replicas_hosts = self.get_group_hosts(group)
        master_hosts = self.get_group_hosts(master_group)

        if len(master_hosts) != 1:
            raise ValueError("Группа %r содержит не один хост" % master_group)

        master_host = master_hosts[0]

        return self.get_hosts_mysql_replicainfo(
            master_host,
            master_group,
            replicas_hosts,
            root_dc
        )

    def get_hosts_mysql_replicainfo(self, master_host, master_group, replicas_hosts, root_dc=None):
        if master_host not in replicas_hosts:
            raise ValueError("Хост %r из %r не является репликой" %
                             (master_host, master_group))

        hosts_info = self.read_hosts(replicas_hosts)

        if root_dc is None:
            root_dc = self.get_current_dc()

        replicas = []

        for host_info in hosts_info:
            host = host_info['fqdn']
            replica = Replica(
                host, host_info['datacenter'], host_info['root_datacenter']
            )
            replica.dc_local = replica.root_dc == root_dc

            if host == master_host:
                replica.priority = 1
                replica.is_master = True

            replicas.append(replica)

        return ReplicaInfo(master_host, replicas)


class LocalReplica(object):
    priority = 0
    dc_local = True

    def __init__(self, host):
        self.host = host

    def __repr__(self):
        return '<%s host=%r>' % (
            self.__class__.__name__,
            self.host,
        )


class Replica(LocalReplica):
    def __init__(self, hostname, dc, root_dc):
        self.hostname = hostname
        self.host = self._hostname_to_ip(hostname)
        self.is_master = False

        self.dc = dc
        self.root_dc = root_dc

        self.dc_local = False

    @staticmethod
    def _hostname_to_ip(hostname):
        try:
            addrinfo = socket.getaddrinfo(hostname, None)
            _, _, _, _, sockaddr = addrinfo[0]
            return sockaddr[0]
        except Exception as ex:
            print(u"Error {} while resolving dns for hostname {}".format(repr(ex), hostname))
            return hostname

    def __repr__(self):
        return '<%s host=%r dc=%r root_dc=%r dc_local=%r master=%r priority=%r>' % (
            self.__class__.__name__,
            self.hostname,
            self.dc,
            self.root_dc,
            self.dc_local,
            self.is_master,
            self.priority,
        )

    def __json__(self):
        return {
            'hostname': self.hostname,
            'host': self.host,
            'dc': self.dc,
            'root_dc': self.root_dc,
            'dc_local': self.dc_local,
            'priority': self.priority,
        }


class ReplicaInfo(object):

    def __init__(self, master=None, replicas=None):
        self.master = master
        self.replicas = replicas or []

    def __repr__(self):
        return '<%s master=%r replicas=%r>' % (
            self.__class__.__name__,
            self.master,
            self.replicas
        )

    def __json__(self):
        return {
            'master': self.master,
            'replicas': self.replicas,
        }


def dc_cache(hosts):
    return {
        'BACKEND': 'travel.avia.library.python.common.utils.memcache_backend.MemcachedCache',
        'LOCATION': ['%s:11211' % host for host in hosts],
        'TIMEOUT': 60,
        'LONG_TIMEOUT': 24 * 60 * 60,
        'OPTIONS': {'server_max_value_length': 128 * 1024 * 1024}
    }


def dc_cache_name(host):
    return 'clustered_cache_' + str(host)


def dc_cache_settings(hosts):
    if not hosts:
        return {}

    return {
        dc_cache_name(host): dc_cache([host])
        for host in hosts
    }
