# -*- coding: utf-8 -*-
import gzip
import logging
from subprocess import Popen, PIPE

from django.db import connection

from travel.avia.library.python.common.utils.progress import PercentageStatus

log = logging.getLogger(__name__)


TIMESTAMP_TABLE_NAME = 'load_dump_timestamp'


def database_exists(db_name):
    cursor = connection.cursor()
    cursor.execute('SHOW DATABASES')
    db_names = [r[0] for r in cursor.fetchall()]
    return db_name in db_names


def list_tables(db_name):
    qn = connection.ops.quote_name
    cursor = connection.cursor()
    cursor.execute('SHOW TABLES FROM {}'.format(qn(db_name)))
    return [r[0] for r in cursor.fetchall()]


def timestamp_table_exists(db_name):
    return TIMESTAMP_TABLE_NAME in list_tables(db_name)


def save_dump_url(db_name, dump_url):
    log.info(u'Записываем dump_url в %s...', db_name)

    qn = connection.ops.quote_name
    cursor = connection.cursor()

    table_name = '{}.{}'.format(qn(db_name), qn(TIMESTAMP_TABLE_NAME))

    cursor.execute('''
        CREATE TABLE {} (`dump_url` varchar (255) PRIMARY KEY)
        DEFAULT CHARSET utf8
    '''.format(table_name))

    cursor.execute('''
        INSERT INTO {} (`dump_url`) VALUES (%s)
    '''.format(table_name), [dump_url])

    connection.connection.commit()


def load_dump_url(db_name):
    if not (database_exists(db_name) and timestamp_table_exists(db_name)):
        return

    qn = connection.ops.quote_name

    cursor = connection.cursor()
    cursor.execute("SELECT `dump_url` FROM {}.{}".format(
        qn(db_name), qn(TIMESTAMP_TABLE_NAME)
    ))

    rows = cursor.fetchall()
    if not rows:
        return

    return rows[0][0]


def purge_database(db_name):
    log.info(u'Пересоздаем %s...', db_name)

    qn = connection.ops.quote_name
    cursor = connection.cursor()

    if database_exists(db_name):
        cursor.execute("DROP DATABASE {}".format(qn(db_name)))

    cursor.execute("CREATE DATABASE {} DEFAULT CHARSET utf8".format(qn(db_name)))


def load_dump_to_database(dump_file, db_name, exclude_tables=()):
    log.info(u'Загружаем файл %s в базу %s...', dump_file.name, db_name)

    log.info(u'Считаем количество строк в файле...')
    line_count = 0
    dump_file.seek(0)
    with gzip.GzipFile(fileobj=dump_file, mode='rb') as f:
        for l in f:
            line_count += 1

    log.info(u'Всего %s строк', line_count)

    db_user = connection.settings_dict['USER']
    db_host = connection.settings_dict['HOST'] or u''
    db_password = connection.settings_dict['PASSWORD']

    process = Popen([
        'mysql',
        '--max-allowed-packet=120M',
        '--net-buffer-length=8M',
        '--user={}'.format(db_user),
        '--host={}'.format(db_host),
        '--password={}'.format(db_password),
        db_name
    ], stdin=PIPE)

    exclude_lines = tuple(
        'INSERT INTO `{}`'.format(t) for t in exclude_tables
    )

    if not exclude_lines:
        log.info(u'Загружаем все без исключений')
    else:
        for line in exclude_lines:
            log.info(u'Исключаем записи начинающиется на "%s"', line)

    status = PercentageStatus(line_count, log)

    dump_file.seek(0)
    with gzip.GzipFile(fileobj=dump_file, mode='rb') as f:
        for l in f:
            if l.startswith('CREATE TABLE'):
                log.info(l[:-2])

            if not l.startswith(exclude_lines):
                process.stdin.write(l)

            status.step(1)

    process.stdin.close()
    retcode = process.wait()

    if retcode != 0:
        raise Exception('Dump was not loaded')
