# -*- coding: utf-8 -*-

import re
import cPickle as pickle
from datetime import date
from xml.parsers.expat import ExpatError
from xml.etree import ElementTree
from itertools import groupby
from cStringIO import StringIO

from django import forms
from django.conf import settings
from django.core.files.storage import FileSystemStorage
from django.db import models
from django.utils.safestring import mark_safe, SafeUnicode
from django.template.loader import render_to_string
from django.core.files.base import File
from django.core.files.storage import get_storage_class
from django.core.urlresolvers import reverse
from django.utils.encoding import smart_str, smart_unicode

from travel.avia.library.python.common.utils.date import make_calendar, RunMask
from travel.avia.library.python.common.utils.text import transliterate


class TransliteratingFileSystemStorage(FileSystemStorage):
    def get_valid_name(self, name):
        valid_name = super(TransliteratingFileSystemStorage, self).get_valid_name(name)
        valid_name = transliterate(valid_name, 'cyr-lat')

        return valid_name


class ReplaceableFileSystemStorage(get_storage_class()):
    def get_available_name(self, name, max_length=None):
        if self.exists(name):
            self.delete(name)

        return name


class TrimmedCharField(models.CharField):
    def pre_save(self, instance, add):
        value = models.CharField.pre_save(self, instance, add)

        if value is not None:
            if isinstance(value, basestring):
                value = value.strip()
        setattr(instance, self.attname, value)
        return value


class TrimmedTextField(models.TextField):
    def pre_save(self, instance, add):
        value = models.TextField.pre_save(self, instance, add)

        if value is not None:
            if isinstance(value, basestring):
                value = value.strip()
        setattr(instance, self.attname, value)
        return value


class PickledObject(str):
    """A subclass of string so it can be told whether a string is
       a pickled object or not (if the object is an instance of this class
       then it must [well, should] be a pickled one)."""
    pass


class PickledObjectField(models.Field):
    def to_python(self, value):
        if isinstance(value, PickledObject):
            # If the value is a definite pickle; and an error is raised in de-pickling
            # it should be allowed to propogate.
            return pickle.loads(str(value))
        else:
            try:
                return pickle.loads(str(value))
            except:
                # If an error was raised, just return the plain value
                return value

    def from_db_value(self, value):
        return self.to_python(value)

    def get_prep_value(self, value):
        if value is not None and not isinstance(value, PickledObject):
            value = PickledObject(pickle.dumps(value))
        return value

    def get_internal_type(self):
        return 'TextField'

    def get_prep_lookup(self, lookup_type, value):
        if lookup_type == 'exact':
            value = self.get_prep_value(value)
            return super(PickledObjectField, self).get_prep_lookup(lookup_type, value)
        elif lookup_type == 'in':
            value = [self.get_prep_value(v) for v in value]
            return super(PickledObjectField, self).get_prep_lookup(lookup_type, value)
        else:
            raise TypeError('Lookup type %s is not supported.' % lookup_type)


class RegExpFormField(forms.CharField):
    def clean(self, value):
        value = super(RegExpFormField, self).clean(value)
        try:
            value = value.strip()
            re.compile(value, re.U)
        except re.error, e:
            raise forms.ValidationError(u'Ошибка в регулярном выражении: %s' % unicode(e))
        return value


class RegExpField(models.CharField):
    def __init__(self, *args, **kwargs):
        super(RegExpField, self).__init__(*args, **kwargs)

    def formfield(self, **kwargs):
        defaults = {'form_class': RegExpFormField}
        defaults.update(kwargs)
        return super(RegExpField, self).formfield(**defaults)


class TextFileWidget(forms.FileInput):
    """
    A FileField Widget that shows its current value if it has one.
    """

    def render(self, name, value, attrs=None):
        output = []
        if value and hasattr(value, "url"):
            output.append('%s <a target="_blank" href="%s">%s</a> <br />' % (u'Скачать:', value.url, value.name))
        output.append(super(TextFileWidget, self).render(name, value, attrs))
        return mark_safe(u''.join(output))


class TextFileFormField(forms.FileField):
    def __init__(self, encoding='utf8', is_xml=False, *args, **kwargs):
        self.encoding = encoding
        self.is_xml = is_xml
        super(TextFileFormField, self).__init__(*args, **kwargs)

    def clean(self, value, initial=None):
        u"""Для FileField в clean передается два аргумента вместо одного"""
        if value is None and initial:
            return initial
        value = super(TextFileFormField, self).clean(value)

        if value is None:
            return None

        try:
            data = value.read().decode(self.encoding)
            value.seek(0)
        except UnicodeDecodeError:
            raise forms.ValidationError(u'Кодировка файла не совпадает с ожидаемой кодировкой: %s' % unicode(self.encoding))
        if self.is_xml:
            try:
                ElementTree.fromstring(data.encode(self.encoding))
            except (ExpatError, ElementTree.ParseError), e:
                raise forms.ValidationError(u'Не валидный xml: %s' % unicode(e))
        return value


class TextFileDescriptor(object):
    def __init__(self, field):
        self.field = field

    def __get__(self, instance=None, owner=None):
        if instance is None:
            raise AttributeError(
                "The '%s' attribute can only be accessed from %s instances."
                % (self.field.name, owner.__name__))

        if instance.__dict__[self.field.name] is None:
            return None

        f = SafeUnicode(instance.__dict__[self.field.name])

        f.encoding = self.field.encoding
        f.url = '/admin/show_file/%s/%s/%s/%s' % (instance._meta.app_label,
                                                  instance.__class__.__name__,
                                                  self.field.name,
                                                  instance.pk)

        f.name = getattr(instance, self.field.name + '_name', None)

        return f

    def __set__(self, instance, value):
        instance.__dict__[self.field.name] = value


class TextFileField(models.Field):
    descriptor_class = TextFileDescriptor

    def __init__(self, *args, **kwargs):

        self.is_xml = kwargs.pop('is_xml', False)
        self.encoding = kwargs.pop('encoding', 'utf8')

        super(TextFileField, self).__init__(*args, **kwargs)

    def get_internal_type(self):
        return "TextField"

    def contribute_to_class(self, cls, name):
        super(TextFileField, self).contribute_to_class(cls, name)
        setattr(cls, self.name, self.descriptor_class(self))

    def save_form_data(self, instance, data):
        setattr(instance, self.name, data)

        if hasattr(data, 'read'):
            if not getattr(instance, self.name + '_name'):
                setattr(instance, self.name + '_name', data.name)

            setattr(instance, self.name, data.read().decode(self.encoding))

    def formfield(self, **kwargs):
        defaults = {
            'form_class': TextFileFormField,
            'max_length': self.max_length,
            'encoding': self.encoding,
            'widget': TextFileWidget,
            'is_xml': self.is_xml}
        # If a file has been provided previously, then the form doesn't require
        # that a new file is provided this time.
        # The code to mark the form field as not required is used by
        # form_for_instance, but can probably be removed once form_for_instance
        # is gone. ModelForm uses a different method to check for an existing file.
        if 'initial' in kwargs:
            defaults['required'] = False
        defaults.update(kwargs)
        return super(TextFileField, self).formfield(**defaults)


class MemoryFile(File):
    def __init__(self, name, content_type, content):
        name = smart_unicode(name)
        self.content_type = content_type
        super(MemoryFile, self).__init__(StringIO(content), name)
        self.size = len(content)

    DELIMITER = '\r\n'

    def to_binary_string(self):
        self.open()
        data = self.read()

        parts = [smart_str(self.name, strings_only=True),
                 smart_str(self.content_type, strings_only=True),
                 data]

        self.open()

        return self.DELIMITER.join(parts)

    def open(self, mode=None):
        self.seek(0)

    def copy(self):
        self.seek(0)
        content = self.read()
        self.seek(0)
        return MemoryFile(self.name, self.content_type, content)

    @classmethod
    def from_file_like_object_string(cls, file_object, content_type):
        return cls(file_object.name, content_type, file_object.read())

    @classmethod
    def from_binary_string(cls, binary_string):
        cls.check_binary_string(binary_string)

        return cls(*binary_string.split(cls.DELIMITER, 2))

    @classmethod
    def check_binary_string(cls, binary_string):
        if not isinstance(binary_string, str):
            raise TypeError('str type must be provided not %s' % type(binary_string))

        if binary_string.count(cls.DELIMITER) < 2:
            raise ValueError('input string does not contain delimiters. Conversion is not posible.')

    def __unicode__(self):
        return self.name


class DatabaseFileWidget(forms.FileInput):
    """
    A FileField Widget that shows its current value if it has one.
    """

    def render(self, name, value, attrs=None):

        output = []
        if value and hasattr(value, "download_url"):
            output.append(u"""<div style="float: left">
%(name)s <a target="_blank" href="%(donwload_url)s">Скачать</a> |
<a href="%(delete_url)s" onclick="return confirm('Вы действительно хотите удалить файл?');">Удалить</a><br />""" %
                          ({
                              'donwload_url': value.download_url,
                              'name': value.name,
                              'delete_url': value.delete_url,
                          }))
        output.append(super(DatabaseFileWidget, self).render(name, value, attrs))

        if value and hasattr(value, "download_url"):
            output.append(u"</div>")

        return mark_safe(u''.join(output))


class DatabaseFileFormField(forms.FileField):
    def __init__(self, content_type='application/octet-stream', *args, **kwargs):
        self.content_type = content_type
        super(DatabaseFileFormField, self).__init__(*args, **kwargs)

    def clean(self, value, initial=None):
        u"""Для FileField в clean передается два аргумента вместо одного"""
        if value is None and initial:
            return initial

        value = super(DatabaseFileFormField, self).clean(value)

        if value is None:
            return None

        value = MemoryFile(value.name, self.content_type, value.read())

        return value


class DatabaseFileDescriptor(object):
    def __init__(self, field):
        self.field = field

    def __get__(self, instance=None, owner=None):
        if instance is None:
            raise AttributeError(
                "The '%s' attribute can only be accessed from %s instances."
                % (self.field.name, owner.__name__))

        if instance.__dict__[self.field.name] is None:
            return None

        memory_file = instance.__dict__[self.field.name]

        if instance.id:
            memory_file.download_url = reverse('admin_download_db_file',
                                               kwargs={'app_label': instance._meta.app_label,
                                                       'model_name': instance.__class__.__name__,
                                                       'pk': instance.id,
                                                       'field_name': self.field.name
                                                       })

            memory_file.delete_url = reverse(
                'admin_delete_db_file',
                kwargs={
                    'app_label': instance._meta.app_label,
                    'model_name': instance.__class__.__name__,
                    'pk': instance.id,
                    'field_name': self.field.name,
                },
            )

        return memory_file

    def __set__(self, instance, value):
        instance.__dict__[self.field.name] = self.field.to_python(value)


class DatabaseFileField(models.Field):

    descriptor_class = DatabaseFileDescriptor

    def contribute_to_class(self, cls, name):
        super(DatabaseFileField, self).contribute_to_class(cls, name)
        setattr(cls, self.name, self.descriptor_class(self))

    def __init__(self, *args, **kwargs):

        self.content_type = kwargs.pop('content_type', 'application/octet-stream')

        super(DatabaseFileField, self).__init__(*args, **kwargs)

    def db_type(self, connection):
        return 'LONGBLOB'

    def to_python(self, value):
        if isinstance(value, MemoryFile):
            return value

        if not value:
            return None

        if isinstance(value, str):
            return MemoryFile.from_binary_string(value)
        elif hasattr(value, 'name') and hasattr(value, 'read'):
            return MemoryFile.from_file_like_object_string(value, self.content_type)
        else:
            raise TypeError("Conversion from %s type not supported" % type(value))

    def get_prep_value(self, value):
        if isinstance(value, MemoryFile):
            return value.to_binary_string()
        else:
            return value

    def get_prep_lookup(self, lookup_type, value):
        # We only handle 'exact'. All others are errors.
        if lookup_type == 'exact':
            return self.get_prep_value(value)
        raise TypeError('Lookup type %r not supported.' % lookup_type)

    def formfield(self, **kwargs):
        defaults = {
            'form_class': DatabaseFileFormField,
            'widget': DatabaseFileWidget,
            'content_type': self.content_type
        }
        # If a file has been provided previously, then the form doesn't require
        # that a new file is provided this time.
        # The code to mark the form field as not required is used by
        # form_for_instance, but can probably be removed once form_for_instance
        # is gone. ModelForm uses a different method to check for an existing file.
        if 'initial' in kwargs:
            defaults['required'] = False
        defaults.update(kwargs)
        return super(DatabaseFileField, self).formfield(**defaults)


class CodeCharField(models.CharField):
    u"""
    Обращает пустую строку в None, чтобы было удобнее обращаться с unique=True
    """

    def pre_save(self, instance, add):
        value = models.CharField.pre_save(self, instance, add)

        if value is not None:
            value = value.strip() or None
            setattr(instance, self.attname, value)
            return value


class ThreadCalendarWidget(forms.Widget):
    class Media:
        css = {
            'all': (settings.STATIC_URL + 'rasp/calendar_widget.css',)
        }
        js = ('//img.yandex.net/y5/1.5-os-c/mega-y5.js', settings.STATIC_URL + 'rasp/calendar_widget.js',)

    def render(self, name, value, attrs):
        if not value:
            value = RunMask.EMPTY_YEAR_DAYS

        mask = RunMask(value, date.today())

        # Нормализуем value (RASP-4298)
        value = str(RunMask(days=mask.dates()))

        data = []

        for _, dates in groupby(mask.all_days, key=lambda d: d.month):
            dates = list(dates)
            first = dates[0].replace(day=1)

            month_data = [None] * 31

            for d in dates:
                month_data[d.day - 1] = mask[d] and '1' or '0'

            data.append((first, month_data))

        pad = (data[0][0].month - 1) % 3

        if pad > 0:
            first_day = data[0][0]
            data = [(date(first_day.year, first_day.month - pad + i, 1), None) for i in range(pad)] + data

        pad_right = 2 - (data[-1][0].month - 1) % 3

        if pad_right > 0:
            first_day = data[-1][0]
            data = data + [(date(first_day.year, first_day.month + i + 1, 1), None) for i in range(pad_right)]

        months = make_calendar(data)

        return render_to_string('admin/block_calendar.html', {'months': months, 'days': value})


class ThreadCalendarField(models.Field):
    """ Класс для поля, хранящего дни хождения нитки """

    def formfield(self, **kwargs):
        defaults = {'widget': ThreadCalendarWidget}
        defaults.update(kwargs)
        return super(ThreadCalendarField, self).formfield(**defaults)

    def get_internal_type(self):
        return 'TextField'
