# -*- coding: utf-8 -*-
import json

import requests
from lxml import etree


class GeoSnippetterClient(object):
    """ https://wiki.yandex-team.ru/maps/dev/core/snippeter/api """

    class Error(Exception):
        pass

    def __init__(self, url, timeout=1):
        self.url = url
        self.timeout = timeout

    def get(self, objects, snippets, lang='ru-RU'):
        objects_json = json.dumps([
            {
                'll': [obj['longitude'], obj['latitude']],
                'rid': obj.get('id', '-1'),
                't': 'geo',
                'k': '',
            }
            for obj in objects
        ])

        try:
            response = requests.get(self.url + '/getsnippets', params={
                'objects': objects_json,
                'snippets': snippets,
                'lang': lang,
                'outformat': 'xml',
            }, timeout=self.timeout)
        except requests.Timeout as ex:
            raise GeoSnippetterClient.Error(ex)

        response.encoding = 'utf8'  # requests неверно угадывает кодировку ответа сниппетовщика
        return response

    def get_near_metro(self, lat, lon):
        """ Получаем информацию про ближайшие к заданной точке станции метро. """
        response = self.get(
            [{'latitude': lat, 'longitude': lon}],
            snippets='masstransit/1.x'
        )
        parser = etree.XMLParser(encoding='utf8')
        xml = etree.fromstring(response.text.encode('utf8'), parser=parser)

        ns = {
            'ns': 'http://maps.yandex.ru/ymaps/1.x',
            'gml': 'http://www.opengis.net/gml',
            'nsm': 'http://maps.yandex.ru/snippets/masstransit/1.x',
        }
        xml_ns = 'http://www.w3.org/XML/1998/namespace'

        stops = []
        stops_path = 'ns:GeoObjectCollection/gml:featureMembers/ns:GeoObject/gml:metaDataProperty/nsm:Stops/nsm:Stop'
        for stop_el in xml.xpath(stops_path, namespaces=ns):
            name_el = stop_el.xpath('nsm:name', namespaces=ns)[0]
            lang = name_el.get('{{{}}}lang'.format(xml_ns))
            color = stop_el.xpath('nsm:Style/nsm:color', namespaces=ns)[0].text
            distance = float(stop_el.xpath('nsm:Distance/nsm:value', namespaces=ns)[0].text)
            coords = stop_el.xpath('gml:Point/gml:pos', namespaces=ns)[0].text.split(' ')
            latitude, longitude = float(coords[1]), float(coords[0])

            stops.append({
                'name': name_el.text,
                'lang': lang,
                'distance': distance,
                'color': color,
                'coords': {
                    'latitude': latitude,
                    'longitude': longitude,
                },
            })

        return stops
