# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import json
import re
from functools import wraps
from xml.etree import ElementTree

from django.http import HttpResponse, HttpResponseBadRequest

from travel.avia.library.python.common.utils.jsonutils import rich_serialize


VALID_CALLBACK_RE = re.compile('^[A-Za-z\d_]*$')


def is_valid_jsonp_callback(callback):
    return VALID_CALLBACK_RE.match(callback) and len(callback) <= 60


def make_jsonp_body(callback, json_body):
    # RASPTICKETS-2043 Возможность обхода SOP через JSONP
    return '/**/%s(%s)' % (callback, json_body)


def jsonp_wrap(request, data, serialize=None):
    callback = request.GET.get('callback', '')

    if not is_valid_jsonp_callback(callback):
        return HttpResponseBadRequest('Invalid callback name', content_type='text/plain')

    serialized = json.dumps(
        data,
        default=serialize or rich_serialize,
        ensure_ascii=False,
        separators=(',', ':')
    )

    if callback:
        response = HttpResponse(make_jsonp_body(callback, serialized),
                                content_type="application/javascript; charset=utf-8")

        response['X-Content-Type-Options'] = 'nosniff'
        response['X-Frame-Options'] = 'SAMEORIGIN'
        response['Content-Disposition'] = 'attachment; filename=json.txt'

        return response

    return HttpResponse(serialized, content_type="application/json; charset=utf-8")


def jsonp_response(func):
    @wraps(func)
    def wrapper(request, *args, **kwargs):
        data = func(request, *args, **kwargs)

        if isinstance(data, HttpResponse):
            return data

        return jsonp_wrap(request, data)

    return wrapper


def xmlresponse(func):
    @wraps(func)
    def wrapper(*args, **kwargs):
        tree = func(*args, **kwargs)

        if isinstance(tree, HttpResponse):
            return tree

        return HttpResponse(ElementTree.tostring(tree, b'UTF-8'), content_type='application/xml')

    return wrapper
