# -*- coding: utf-8 -*-
import codecs
import logging
import multiprocessing
import os.path
import sys
import threading
import traceback
from datetime import datetime
import Queue as queue  # Python 2.

from django.core.files.utils import FileProxyMixin

from .exceptions import SimpleUnicodeException


# !!!! Logging hack
logging._srcfile = os.path.normcase(logging.log.func_code.co_filename)


def patchedHandleError(self, record):
    """
    Handle errors which occur during an emit() call.

    This method should be called from handlers when an exception is
    encountered during an emit() call. If raiseExceptions is false,
    exceptions get silently ignored. This is what is mostly wanted
    for a logging system - most users will not care about errors in
    the logging system, they are more interested in application errors.
    You could, however, replace this with a custom handler if you wish.
    The record which was being processed is passed in to this method.
    """
    if logging.raiseExceptions and sys.stderr:  # see issue 13807
        ei = sys.exc_info()
        try:
            traceback.print_exception(ei[0], ei[1], ei[2],
                                      None, sys.stderr)
            traceback.print_stack(None, None, sys.stderr)
            sys.stderr.write('%s Logged from file %s, line %s\n' % (
                             datetime.now(), record.filename, record.lineno))
        except IOError:
            pass    # see issue 5971
        finally:
            del ei


logging.Handler.handleError = patchedHandleError


class LogError(SimpleUnicodeException):
    pass


class WatchedFileHandler(logging.FileHandler):
    def _open(self):
        self.encoding = self.encoding or 'utf-8'

        return WatchedFile(self.baseFilename, mode=self.mode, encoding=self.encoding)


class WatchedFile(FileProxyMixin):
    """Файл, которые переоткрывается при его переименовании или удалении"""

    def __init__(self, filename, mode='a', encoding='utf-8'):
        self.file = None
        self.mode = mode
        self._encoding = encoding

        self.filename = filename
        self.dirname = os.path.dirname(self.filename)

        if not os.path.exists(self.filename):
            self.dev, self.ino = None, None
        else:
            stat = os.stat(self.filename)
            self.dev, self.ino = stat.st_dev, stat.st_ino

        self.open()

    def open(self):
        if not os.path.exists(self.dirname):
            os.makedirs(self.dirname)

        self.close()

        self.file = codecs.open(self.filename, self.mode, encoding=self._encoding)

        st = os.stat(self.filename)
        self.dev, self.ino = st.st_dev, st.st_ino

    def reopen(self):
        self.open()

    def flush(self):
        self.file and self.file.flush()

    def write(self, msg):
        if not os.path.exists(self.filename):
            changed = True

        else:
            st = os.stat(self.filename)
            the_same = (st.st_dev == self.dev) and (st.st_ino == self.ino)

            changed = not the_same

        if changed:
            self.reopen()

        if isinstance(msg, unicode):
            self.file.write(msg)
        else:
            self.file.write(unicode(msg, 'utf-8', 'replace'))

    def close(self):
        if not self.file:
            return

        if not self.file.closed:
            self.file.flush()
            self.file.close()


class FileTreeHandler(logging.Handler):
    """
    Обработчик, создающий дерево из обрабатываемых логов
    """

    SUFFIX = '.log'

    def __init__(self, base_path):
        """
        Initialize the handler.
        """
        logging.Handler.__init__(self)
        self.base_path = base_path
        self.streams = {}

    def get_stream(self, name):
        if name in self.streams:
            return self.streams[name]

        parts = name.split('.')

        path = os.path.join(self.base_path, *parts) + self.SUFFIX

        stream = self.streams[name] = WatchedFile(path)

        return stream

    def flush(self):
        """
        Flushes the streams.
        """

        for stream in self.streams.values():
            stream.flush()

    def emit(self, record):
        """
        Emit a record.

        If a formatter is specified, it is used to format the record.
        The record is then written to the stream with a trailing newline.
        """

        try:
            msg = self.format(record)

            stream = self.get_stream(record.name)

            stream.write("%s\n" % msg)

            stream.flush()

        except (KeyboardInterrupt, SystemExit):
            raise
        except:
            self.handleError(record)


class ArcadiaFileTreeHandler(FileTreeHandler):
    """
    Обработчик, создающий дерево из обрабатываемых логов
    с возможностью обрезать ненужную часть пути до файлов,
    например 'travel.avia.backend'
    """

    def __init__(self, base_path, remove_prefix=None):
        super(ArcadiaFileTreeHandler, self).__init__(base_path)
        self.remove_prefix = remove_prefix
        self.base_path = base_path
        self.streams = {}

    def get_stream(self, name):
        if name in self.streams:
            return self.streams[name]

        if self.remove_prefix and name.startswith(self.remove_prefix):
            cut_length = len(self.remove_prefix) if self.remove_prefix.endswith('.') else len(self.remove_prefix) + 1
            parts = name[cut_length:].split('.')
        else:
            parts = name.split('.')

        path = os.path.join(self.base_path, *parts) + self.SUFFIX

        stream = self.streams[name] = WatchedFile(path)

        return stream


class ExcludeFromTree(logging.Filter):
    """
    Исключает из корневого лога дочерние логи.

    >>> logging_config = {
    >>>     'filters': {
    >>>         'exclude_from_tree': {
    >>>             '()': 'travel.avia.library.python.common.utils.logs.ExcludeFromTree',
    >>>             'starts_with_names': ('special.',),
    >>>             'exact_names': ('__main__',),
    >>>             'ignore_names': ('special.pypy',)
    >>>         }
    >>>     }
    >>>     'handlers': {
    >>>         'tree_handler': {
    >>>             'level': 'DEBUG',
    >>>             'class': 'travel.avia.library.python.common.utils.logs.FileTreeHandler',
    >>>             'base_path': os.path.join(log_path, 'tree'),
    >>>             'formatter': 'verbose',
    >>>             'filters': ['exclude_from_tree']
    >>>         }
    >>>     }
    >>> }

    Не будут логироваться все логи начинающиеся со special. и __main__.
    special.pypy при это продолжит логироваться.
    """

    def __init__(self, exact_names=tuple(), starts_with_names=tuple(), ignore_names=tuple()):
        self.exact_names = exact_names
        self.starts_with_names = starts_with_names
        self.ignore_names = ignore_names

    def filter(self, record):
        if record.name in self.exact_names:
            return False

        if record.name in self.ignore_names:
            return True

        if record.name.startswith(self.starts_with_names):
            return False

        return True


class MultiProcessingHandler(logging.Handler):
    def __init__(self, name='test', sub_handler=None):
        super(MultiProcessingHandler, self).__init__()

        if sub_handler is None:
            sub_handler = logging.StreamHandler()
        self.sub_handler = sub_handler

        self.queue = multiprocessing.Queue(-1)
        self._is_closed = False
        # The thread handles receiving records asynchronously.
        self._receive_thread = threading.Thread(target=self._receive, name=name)
        self._receive_thread.daemon = True
        self._receive_thread.start()

    def setLevel(self, level):
        self.sub_handler.setLevel(level)
        super(MultiProcessingHandler, self).setLevel(level)

    def setFormatter(self, fmt):
        self.sub_handler.setFormatter(fmt)
        super(MultiProcessingHandler, self).setFormatter(fmt)

    def _receive(self):
        while not (self._is_closed and self.queue.empty()):
            try:
                record = self.queue.get(timeout=0.2)
                self.sub_handler.emit(record)
            except (KeyboardInterrupt, SystemExit):
                raise
            except EOFError:
                break
            except queue.Empty:
                pass  # This periodically checks if the logger is closed.
            except:
                traceback.print_exc(file=sys.stderr)

        self.queue.close()
        self.queue.join_thread()

    def _send(self, s):
        self.queue.put_nowait(s)

    def _format_record(self, record):
        # ensure that exc_info and args
        # have been stringified. Removes any chance of
        # unpickleable things inside and possibly reduces
        # message size sent over the pipe.
        if record.args:
            record.msg = record.msg % record.args
            record.args = None
        if record.exc_info:
            self.format(record)
            record.exc_info = None

        return record

    def emit(self, record):
        try:
            msg = self._format_record(record)
            self._send(msg)
        except (KeyboardInterrupt, SystemExit):
            raise
        except:
            self.handleError(record)

    def close(self):
        if not self._is_closed:
            self._is_closed = True
            self._receive_thread.join(5.0)  # Waits for receive queue to empty.

            self.sub_handler.close()
            super(MultiProcessingHandler, self).close()


class MultiProcessingWatchedFileHandler(MultiProcessingHandler):
    def __init__(self, filename):
        super(MultiProcessingWatchedFileHandler, self).__init__(
            sub_handler=WatchedFileHandler(filename)
        )


class MultiProcessingFileTreeHandler(MultiProcessingHandler):
    def __init__(self, base_path):
        super(MultiProcessingFileTreeHandler, self).__init__(
            sub_handler=FileTreeHandler(base_path)
        )
