# -*- coding: utf-8 -*-
from django import forms
from django.db import models
from django.utils.translation import ugettext_lazy as _
from lxml import etree


BYTES_IN_MEGABYTE = 2**20


class CleanSizeCheckerMixin(object):
    MAX_SIZE = BYTES_IN_MEGABYTE

    def clean(self, *args, **kwargs):
        data = super(CleanSizeCheckerMixin, self).clean(*args, **kwargs)
        try:
            data.file
        except IOError:
            return data
        if not data.file:
            return data
        if self.MAX_SIZE is not None and _safe_get_size(data.file) > self.MAX_SIZE:
            raise forms.ValidationError(
                _(u'Размер файла %s превышает допустимый лимит %.2f МБ') %
                (data.name, self.MAX_SIZE / BYTES_IN_MEGABYTE)
            )
        return data


def _safe_get_size(container):
    if hasattr(container, 'size'):
        return container.size
    elif hasattr(container, 'getvalue'):
        return len(container.getvalue())
    else:
        raise NotImplementedError


class CleanSvgMixin(CleanSizeCheckerMixin):
    def clean(self, *args, **kwargs):
        data = super(CleanSvgMixin, self).clean(*args, **kwargs)
        try:
            data.file
        except IOError:
            return data
        if not data.file:
            return data
        try:
            etree.parse(data.file)
        except IOError:
            raise forms.ValidationError(_(u'Файл не найден %s') % data.name)
        except Exception as e:
            raise forms.ValidationError(_(u'Формат файла не поддерживается: %s' % e))
        data.file.seek(0)  # etree drains file so we need to rewind it
        return data


IMAGES_FORMATS_EXTENSIONS = {
    'gif': 'gif',
    'png': 'png',
    'jpeg': 'jpg',
}


class CleanImageMixin(CleanSizeCheckerMixin):
    def clean(self, *args, **kwargs):
        data = super(CleanImageMixin, self).clean(*args, **kwargs)
        try:
            data.file
        except IOError:
            return data
        if not data.file:
            return data
        # Try to import PIL in either of the two ways it can end up installed.
        try:
            from PIL import Image
        except ImportError:
            import Image
        try:
            image = Image.open(data.file)
        except IOError:
            raise forms.ValidationError(_(u'Файл не найден %s') % data.name)
        image_format = image.format.lower()
        try:
            self.file_ext = IMAGES_FORMATS_EXTENSIONS[image_format]
        except KeyError:
            raise forms.ValidationError(_(u'Формат файла не поддерживается: %s' % image_format))
        return data


class CustomImageField(CleanImageMixin, models.ImageField):
    pass


class SvgImageField(CleanSvgMixin, models.FileField):
    pass
