# -*- coding: utf-8 -*-

from django.core.cache.backends import memcached
from django.utils.functional import cached_property


class MemcachedCache(memcached.MemcachedCache):
    """
        Бэкенд для memcache, передающий OPTIONS в качестве настроек клиента и
        MIN_COMPRESS_LEN для сжатия длинных значений.
    """

    def __init__(self, server, params):
        super(MemcachedCache, self).__init__(server, params)

        self._min_compress_len = params.get('MIN_COMPRESS_LEN', 0)

    @cached_property
    def _cache(self):
        kwargs = self._options or {}
        return self._lib.Client(self._servers, **kwargs)

    def add(self, key, value, timeout=0, version=None):
        key = self.make_key(key, version=version)
        return self._cache.add(key, value, self.get_backend_timeout(timeout),
                               self._min_compress_len)

    def set(self, key, value, timeout=0, version=None):
        key = self.make_key(key, version=version)
        self._cache.set(key, value, self.get_backend_timeout(timeout),
                        self._min_compress_len)

    def set_many(self, data, timeout=0, version=None):
        safe_data = {}
        for key, value in data.items():
            key = self.make_key(key, version=version)
            safe_data[key] = value
        self._cache.set_multi(safe_data, self.get_backend_timeout(timeout),
                              min_compress_len=self._min_compress_len)
