# -*- coding: utf-8 -*-

import logging
import warnings

import MySQLdb
from MySQLdb.constants import FIELD_TYPE, FLAG
from django.conf import settings
from django.utils.functional import cached_property
from django.utils.safestring import SafeBytes, SafeText
from django.db.backends.mysql.base import (
    DatabaseWrapper as MysqlWrapper, django_conversions,
    Database, CLIENT,
)
from django.db.utils import DEFAULT_DB_ALIAS, DatabaseErrorWrapper as DjangoDBErrorWrapper, OperationalError
from django.utils import six
from django.utils.encoding import force_str
from retrying import retry

from travel.avia.library.python.common.utils import connectdb, replicastatestore


log = logging.getLogger(__name__)

try:
    import pymysql

except ImportError:
    mysqldb_is_pymysql = False

else:
    mysqldb_is_pymysql = (MySQLdb is pymysql)

warnings.simplefilter("ignore", MySQLdb.Warning)

rasp_conversions = django_conversions.copy()


def string_decode(s):
    return s.decode('utf-8')


rasp_conversions.update({
    FIELD_TYPE.STRING: [(None, string_decode)],
    FIELD_TYPE.VAR_STRING: [(None, string_decode)],
    FIELD_TYPE.VARCHAR: [(None, string_decode)],
    FIELD_TYPE.BLOB: ((FLAG.BINARY, None), (None, string_decode))
})

TIMEOUT_OPTIONS = {
    'connect_timeout': settings.MYSQL_CONNECT_TIMEOUT,
    'read_timeout': settings.MYSQL_READ_TIMEOUT,
    'write_timeout': settings.MYSQL_WRITE_TIMEOUT,
}

CONNECT_OPTIONS = {
    'charset': 'utf8',
    'use_unicode': False,
    'local_infile': True,
    # We need the number of potentially affected rows after an
    # "UPDATE", not the number of changed rows.
    'client_flag': CLIENT.FOUND_ROWS
}

if mysqldb_is_pymysql:
    # pymysql не поддерживает local_infile, read_timeout, write_timeout и
    # списки в conv но, там где он используется, это не нужно
    del (
        CONNECT_OPTIONS['local_infile'],
        TIMEOUT_OPTIONS['read_timeout'],
        TIMEOUT_OPTIONS['write_timeout']
    )

    rasp_conversions.update({
        FIELD_TYPE.STRING: string_decode,
        FIELD_TYPE.VAR_STRING: string_decode,
        FIELD_TYPE.VARCHAR: string_decode,
        FIELD_TYPE.BLOB: string_decode,
    })


class DatabaseWrapper(MysqlWrapper):
    base_aliases = ['work_db', 'service_db', 'migration_db']
    """Класс для использования реплик mysql"""

    db_names = {
        alias: settings.DATABASES[alias]['NAME']
        for alias in settings.DATABASES
    }
    db_names[DEFAULT_DB_ALIAS] = settings.DATABASES['default']['NAME']

    cache_tag = None

    def __init__(self, *args, **kwargs):
        super(DatabaseWrapper, self).__init__(*args, **kwargs)
        self.alias = args[1] if len(args) > 1 else kwargs.get('alias', DEFAULT_DB_ALIAS)
        if not self.host_defined and self.is_master_prefered:
            self.define_host()

        self._connected_to_db_name = None  # Текущая база, к которой создан коннект

    @retry(wait_fixed=50, stop_max_attempt_number=3)
    def _get_new_connection(self, conn_params):
        if not self.host_defined:
            conn_params.update(TIMEOUT_OPTIONS)
            replicas = self.settings_dict['REPLICAS']
            conn = connectdb.connect(replicas, replicastatestore.get(self.settings_dict), **conn_params)
        else:
            conn = Database.connect(**conn_params)

        conn.encoders[SafeText] = conn.encoders[six.text_type]
        conn.encoders[SafeBytes] = conn.encoders[bytes]

        self._connected_to_db_name = conn_params['db']

        return conn

    def get_new_connection(self, conn_params):
        try:
            return self._get_new_connection(conn_params)
        except Exception as e:
            log.exception('Failed to connect to MySql %r', e)
            raise

    def get_connection_params(self):
        self.sync_if_needed()

        conn_params = {
            'conv': rasp_conversions,
            'charset': 'utf8',
        }

        settings_dict = self.settings_dict
        if settings_dict['USER']:
            conn_params['user'] = settings_dict['USER']
        if settings_dict['PASSWORD']:
            conn_params['passwd'] = force_str(settings_dict['PASSWORD'])

        if self.host_defined:
            if settings_dict['HOST'].startswith('/'):
                conn_params['unix_socket'] = settings_dict['HOST']
            elif settings_dict['HOST']:
                conn_params['host'] = settings_dict['HOST']

        if settings_dict['PORT']:
            conn_params['port'] = int(settings_dict['PORT'])
        # We need the number of potentially affected rows after an
        # "UPDATE", not the number of changed rows.
        conn_params['client_flag'] = CLIENT.FOUND_ROWS
        conn_params.update(settings_dict['OPTIONS'])

        conn_params.update(CONNECT_OPTIONS)
        conn_params['db'] = self.db_names[self.alias]
        return conn_params

    @cached_property
    def wrap_database_errors(self):
        """
        Context manager and decorator that re-throws backend-specific database
        exceptions using Django's common wrappers.
        """
        return DatabaseErrorWrapper(self)

    def define_host(self):
        conn_params = self.get_connection_params()
        state_store = replicastatestore.get(self.settings_dict)
        connectdb.actualize_replicas_info(
            replicas=self.settings_dict['REPLICAS'],
            state_store=state_store,
            kwargs=conn_params,
        )
        for replica in self.settings_dict['REPLICAS']:
            if replica.is_master:
                log.info('Redefine database host to %s', replica.host)
                self.settings_dict['HOST'] = replica.host
                break

    def init_connection_state(self):
        with self.cursor() as cursor:
            # SQL_AUTO_IS_NULL in MySQL controls whether an AUTO_INCREMENT column
            # on a recently-inserted row will return when the field is tested for
            # NULL.  Disabling this value brings this aspect of MySQL in line with
            # SQL standards.
            cursor.execute('SET SQL_AUTO_IS_NULL = 0')

            # https://dev.mysql.com/doc/refman/5.7/en/server-system-variables.html#sysvar_storage_engine
            # This variable is deprecated and was removed in MySQL 5.7.5. Use default_storage_engine instead.
            # cursor.execute('SET storage_engine=InnoDB')

    def ensure_connection(self):
        self.sync_if_needed()

        if (
            self.connection is None or
            not self.is_usable() or
            self._connected_to_db_name != self.db_names[self.alias]
        ):
            with self.wrap_database_errors:
                self.close()
                self.connect()

    @property
    def host_defined(self):
        return bool(self.settings_dict.get('HOST'))

    @property
    def is_master_prefered(self):
        return bool(self.settings_dict.get('USE_MASTER_DB'))

    @classmethod
    def get_db_name(cls, db_alias=None):
        cls.sync_if_needed()

        db_alias = db_alias or DEFAULT_DB_ALIAS

        return cls.db_names.get(db_alias)

    @classmethod
    def is_synced(cls):
        """Faked"""
        # return settings.INSTANCE_ROLE.db_alias in cls.db_names
        return True

    @classmethod
    def sync_if_needed(cls):
        if not cls.is_synced():
            cls.sync_db()

    @classmethod
    def sync_db(cls):
        """Сверяет настройки с эксплуатационной базой"""
        raise Exception('Do not use mysql_switcher.base.DatabaseWrapper')

    @classmethod
    def update_cache_root(cls, new_db_name, new_cache_tag):
        raise Exception('Do not use mysql_switcher.base.DatabaseWrapper')

    @classmethod
    def update_db_names(cls, conf):
        raise Exception('Do not use mysql_switcher.base.DatabaseWrapper')


class DatabaseErrorWrapper(DjangoDBErrorWrapper):
    def __exit__(self, exc_type, exc_value, traceback):
        try:
            super(DatabaseErrorWrapper, self).__exit__(exc_type, exc_value, traceback)
        except OperationalError, e:
            if is_read_only_error(e):
                redefine_host_if_needed(self.wrapper)
            raise


def is_read_only_error(exc):
    message = 'The MySQL server is running with the --read-only option so it cannot execute this statement'
    return isinstance(exc, OperationalError) and exc.args[1] == message


def redefine_host_if_needed(db_wrapper):
    # type: (DatabaseWrapper) -> None
    db_wrapper.close()
    if db_wrapper.is_master_prefered:
        db_wrapper.define_host()
