# -*- coding: utf-8 -*-

import json
import re

from urlparse import urlparse, urlunparse

from django.core.urlresolvers import resolve, Resolver404
from django.http import QueryDict

from travel.avia.library.python.common.xgettext.i18n import xgettext


schema_simple_commands = {
    '-': 'rl',
    '|': 'tb',
    '7': 'rb_c',
    '9': 'bl_c',
    '5': 'trbl',
    '1': 'tr_c',
    '3': 'tl_c',
    '6': 'trb',
    '4': 'tbl',
    '8': 'rbl',
    '2': 'trl',
    't': 'b_f',
    'b': 't_f',
    'l': 'r_f',
    'r': 'l_f',
}

schema_simple_commands_by_item = dict((item, cmd) for cmd, item in schema_simple_commands.items())


schema_commands = {
    '0': 'tb_s',
    '100': 'rl_s',
    '8': 'b_s',
    '2': 't_s',
    '108': 'brl_s',
    '102': 'trl_s',
    '4': 'tbl_s',
    '1': 'tr_s',
    '3': 'tl_s',
    '7': 'rb_s',
    '9': 'bl_s',
    '5': 'trbl_s',
    '6': 'trb_s',
    '110': 'rl_s1',
    '10': 'tb_s1',
    '18': 'b_s1',
    '12': 't_s1',
    '118': 'brl_s1',
    '112': 'trl_s1',
    '14': 'tbl_s1',
    '11': 'tr_s1',
    '13': 'tl_s1',
    '17': 'rb_s1',
    '19': 'bl_s1',
    '15': 'trbl_s1',
    '16': 'trb_s1',
    '120': 'rl_m1',
    '20': 'tb_m1',
    '28': 'b_m1',
    '22': 't_m1',
    '128': 'brl_m1',
    '122': 'trl_m1',
    '24': 'tbl_m1',
    '21': 'tr_m1',
    '23': 'tl_m1',
    '27': 'rb_m1',
    '29': 'bl_m1',
    '25': 'trbl_m1',
    '26': 'trb_m1',
    '130': 'rl_m',
    '30': 'tb_m',
    '38': 'b_m',
    '32': 't_m',
    '138': 'brl_m',
    '132': 'trl_m',
    '34': 'tbl_m',
    '31': 'tr_m',
    '33': 'tl_m',
    '37': 'rb_m',
    '39': 'bl_m',
    '35': 'trbl_m',
    '36': 'trb_m',

    '84': 'r_f',
    '86': 'l_f',
    '82': 't_f',
    '88': 'b_f',
}

schema_commands_by_item = dict((item, cmd) for cmd, item in schema_commands.items())

schema_cmd_re = re.compile(ur'^(?:(?P<code>\d+)\^)?(?P<label>.*?)(?:#(?P<subdir>.*?))?(?:&(?P<link>.*?))?'
                           ur'(?:@(?P<metro>.*?))?(?:\$(?P<busstations>.*?))?$',  # RASP-8930
                           re.U)


def build_schema_cmd_str(code, label=None, subdir=None, href=None, metro=None, busstations=[]):
    # if not label:
    #     raise ValueError('empty label')

    if code == '0':
        code = ''

    cmd = (("%s^" % code) if code else "") + (label or '')

    if href:
        if not href.isdigit():
            # Не esr-код, нужно парсить
            o = urlparse(href)

            try:
                view, args, kwargs = resolve(o.path)
            except Resolver404:
                view = None

            # Так вычисляем, что это ссылка на станцию
            if getattr(view, 'station_view', None):
                q = QueryDict(o.query)

                from travel.avia.library.python.common.models.geo import StationCode

                esr_code = StationCode.objects.get(station__id=kwargs['station_id'], system__code='esr').code

                href = esr_code

                subdir = q.get('subdir')
            else:
                # Иначе просто ссылка в сервисе
                href = urlunparse((None, None, o.path, o.params, o.query, o.fragment))

    if subdir:
        cmd += '#%s' % subdir

    if href:
        cmd += '&%s' % href

    if metro:
        cmd += '@%s' % metro

    if busstations:
        cmd += '$%s' % ','.join(busstations)

    return cmd


class ScheduleLink(object):
    def __init__(self, esr_code, subdir):
        self.esr_code = esr_code
        self.subdir = subdir


def parse_schema_command(cmd):
    if not cmd:
        return None, None, None, None, []

    if cmd in schema_simple_commands:
        item = schema_simple_commands[cmd]

        return item, None, None, None, []

    match = schema_cmd_re.match(cmd)

    if not match:
        raise ValueError('Unknown format of commands %r' % cmd)

    data = match.groupdict('')

    code = data['code'] or '0'

    try:
        item = schema_commands[code]
    except KeyError:
        raise ValueError('Unknown command %r' % code)

    href = data['link']

    if href.isdigit():
        if len(href) > 3:
            href = '0' * (6 - len(href)) + href
        href = ScheduleLink(href, data['subdir'])

    label = data['label']

    metro = data['metro']

    busstations = map(int, data['busstations'].split(',')) if data['busstations'] else []

    return item, href, label, metro, busstations


def build_schema_command(cmd):
    if not cmd.get('item'):
        return ""

    if not cmd.get('label') and not cmd.get('href') and cmd.get('item') in schema_simple_commands_by_item:
        return schema_simple_commands_by_item[cmd.get('item')]

    return build_schema_cmd_str(
        code=schema_commands_by_item[cmd.get('item')],
        href=cmd.get('href'),
        label=cmd.get('label'),
        metro=cmd.get('metro', None),
        busstations=cmd.get('busstations', []),
    )


def initial_schema(direction):
    rows = [
        [(None, None, None, None, [],)] * 15 +
        [('tb_s', dm.station.get_schedule_url(type_='suburban', direction=direction.code), dm.station.title, None, [])] +
        [(None, None, None, None, [],)] * 15
        for dm in direction.externaldirectionmarker_set.all()
    ]
    return [['Z', rows]]


def parse_schema(schema_file, direction_code, edit=False):

    zones = []

    for line in schema_file.split('\n'):
        commands = [f.strip() for f in line.split(';')][:32]

        if not any(commands):
            continue

        num = commands[0]

        if num or not zones:
            rows = []
            zones.append((num or 'Z', rows))

        row = []

        rows.append(row)

        for cmd in commands[1:]:
            row.append(parse_schema_command(cmd))

    esr_codes = [
        href.esr_code
        for _num, _rows in zones
        for _row in _rows
        for item, href, label, metro, busstations in _row
        if isinstance(href, ScheduleLink)
    ]

    from travel.avia.library.python.common.models.geo import StationCode
    from travel.avia.library.python.common.models.geo import DirectionFromTranslate

    stations = StationCode.objects.filter(
        code__in=esr_codes, system__code='esr',
    ).select_related('station', 'station__t_type', 'station__station_type').order_by()

    mapping = dict((sc.code, sc.station) for sc in stations)

    # Ссылки на станции по esr-кодам
    for num, rows in zones:
        for row in rows:
            for i, item in enumerate(row):
                item, href, label, metro, busstations = item

                if href and isinstance(href, ScheduleLink):
                    link, href = href, None

                    station = mapping.get(link.esr_code)

                    if station:
                        label = station.L_title()

                        if station.t_type.code == 'plane' and station.station_type.id == 9:
                            label = xgettext(u'Аэропорт <airport-title/>', airport_title=label)

                        if not station.hidden:
                            href = station.get_schedule_url(type_='suburban', direction=direction_code, subdir=link.subdir)

                elif label:
                    label = DirectionFromTranslate.get(label)

                row[i] = item, href, label, metro, busstations

    return zones


def build_schema(schema_json):
    parsed = json.loads(schema_json)

    schema_file_lines = []

    for zone in parsed:
        first = True

        for row in zone['rows']:
            # Номер зоны проставляется в первой строке зоны
            num = zone['num'] if first else ''

            commands = [num]

            for cmd in row:
                commands.append(build_schema_command(cmd))

            schema_file_lines.append(";".join(commands))

            first = False

    return "\n".join(schema_file_lines)
