# -*- coding: utf-8 -*-

import json
from tempfile import NamedTemporaryFile
import logging

from django.conf import settings
from django.utils.encoding import force_unicode

from django_tanker import api

log = logging.getLogger(__name__)


def tanker_serialize(obj):
    if hasattr(obj, '__json__'):
        return obj.__json__()

    raise TypeError(repr(obj) + ' is not JSON serializable')


def tanker_json_dumps(obj):
    return unicode(json.dumps(
        obj, ensure_ascii=False, indent=4,
        default=tanker_serialize,
        separators=(',', ': '), sort_keys=True,
    ))


def strip_tjson_keyset(keyset):
    for key_translations in keyset.values():
        for translation in key_translations['translations'].values():
            for form_key in translation.keys():
                if form_key.startswith(u'form'):
                    translation[form_key] = translation[form_key].strip()

    return keyset


class Tanker(object):
    def __init__(
        self, tanker='testing', token=None, dry_run=False, mode=None,
        project=None, branch='master',
    ):
        self.tanker = api.Tanker(
            project, api.URLS[tanker], token=token or settings.TANKER_TOKEN,
            dry_run=dry_run, include_unapproved=True,
        )
        self.mode = mode
        self.branch = branch

    def list(self):
        return self.tanker.list()

    def get_mode(self, keyset, languages):
        if self.mode:
            return self.mode

        ukeyset = force_unicode(keyset)
        ulanguages = ','.join(force_unicode(lang) for lang in languages)

        if keyset not in self.list():
            log.info('Creating %s[%s]', ukeyset, ulanguages)
            return 'create'

        else:
            log.info('Updating %s[%s]', ukeyset, ulanguages)
            return 'replace'

    def upload(self, keyset, keys, languages=['ru']):
        data = tanker_json_dumps({
            'keysets': {
                keyset: {
                    'keys': keys
                }
            }
        })

        with NamedTemporaryFile(suffix=".tjson") as f:
            f.write(data.encode('utf-8'))
            f.flush()

            mode = self.get_mode(keyset, languages)

            log.info('%d keys', len(keys))

            return self.tanker.upload(
                keyset, ','.join(languages), file_info=f.name, mode=mode,
                key_not_language=True, file_format='tjson', branch=self.branch
            )

    def upload_po(self, keyset, pos):
        mode = self.get_mode(keyset, pos.keys())

        for lang, path in pos.items():
            return self.tanker.upload(
                keyset, lang, path, mode=mode, key_not_language=True,
                file_format='po', branch=self.branch,
            )

    def download(self, keyset, languages):
        tjson = self.tanker.download(
            keyset, ','.join(languages),
            key_not_language=True, file_format='tjson', ref=self.branch,
        )

        data = json.loads(tjson)

        if keyset in data['keysets']:
            return data['keysets'][keyset]['keys']

        return {}

    def download_po(self, keyset, pos, dry_run=False):
        for lang, path in pos.items():
            self._log(keyset, lang, path)

            po = self.tanker.download(
                keyset, lang, key_not_language=False, file_format='po',
                ref=self.branch,
            )

            self._write(path, po, dry_run)

    def download_bem(self, block, lang, path, dry_run=False, prefix='blocks-desktop'):
        keyset = "%s:%s" % (prefix, block)

        self._log(keyset, lang, path)

        text = self.tanker.download(
            keyset, lang, key_not_language=True, file_format='json',
            ref=self.branch,
        )

        data = json.loads(text)

        text = 'module.exports = %s;\n' % json.dumps({block: data[lang]}, indent=4, ensure_ascii=False).encode('utf-8')

        self._write(path, text, dry_run)

    def download_tjson(self, keyset, languages, path, dry_run=False, strip_forms=False):
        self._log(keyset, languages, path)

        keyset = self.download(keyset, languages)

        if strip_forms:
            keyset = strip_tjson_keyset(keyset)

        text = tanker_json_dumps(keyset).encode('utf-8')

        self._write(path, text, dry_run)

    def download_lang_keyset_key_json(self, keyset_name, languages, path, dry_run=False):
        self._log(keyset_name, languages, path)

        keyset_raw = self.tanker.download(
            keyset_name, ','.join(languages),
            key_not_language=True, file_format='json', ref=self.branch,
        )
        keyset = json.loads(keyset_raw)

        formatted_keyset = {}
        for lang in keyset:
            formatted_keyset[lang] = {
                keyset_name: {
                    k: (v.strip() if isinstance(v, basestring) else v)
                    for k, v in keyset[lang].items()
                }
            }

        out_data = tanker_json_dumps(formatted_keyset)

        self._write(path, out_data.encode('utf-8'), dry_run)

    def _write(self, path, data, dry_run):
        if dry_run:
            return

        with open(path, 'w') as f:
            f.write(data)

    def _log(self, keyset, lang, path):
        log.info('Downloading %s[%s] to: %s', tuple(map(force_unicode, (keyset, lang, path))))
