# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import re


NBSP = u'\xa0'


def _compile_char_map(char_map):
    map_re = re.compile('|'.join(re.escape(char) for char in char_map.keys()))

    def repl(match):
        return char_map[match.group()]

    def mapper(text):
        return map_re.sub(repl, text)

    return mapper


TRANSLIT_TABLES = {
    'cyr-lat': {
        'а': 'a', 'б': 'b', 'в': 'v', 'г': 'g', 'д': 'd', 'е': 'e',
        'ж': 'zh', 'з': 'z', 'и': 'i', 'й': 'j', 'к': 'k', 'л': 'l',
        'м': 'm', 'н': 'n', 'о': 'o', 'п': 'p', 'р': 'r', 'с': 's',
        'т': 't', 'у': 'u', 'ф': 'f', 'х': 'x', 'ц': 'cz', 'ч': 'ch',
        'ш': 'sh', 'щ': 'shh', 'ъ': 'qd', 'ы': 'qi', 'ь': 'qt', 'э': 'ye',
        'ю': 'yu', 'я': 'ya', 'ё': 'yo'
    },

    # таблица сделана в основном расчете на транслитерированный поиск, возможно требуется
    # коррекция для более широкого применения
    'lat-cyr': {
        'a': 'а', 'b': 'б', 'c': 'ц', 'd': 'д', 'e': 'е', 'f': 'ф',
        'g': 'г', 'h': 'х', 'i': 'и', 'j': 'й', 'k': 'к', 'l': 'л',
        'm': 'м', 'n': 'н', 'o': 'о', 'p': 'п', 'q': 'я', 'r': 'р',
        's': 'с', 't': 'т', 'u': 'у', 'v': 'в', 'w': 'в', 'x': 'х',
        'y': 'ы', 'z': 'з'
    },
}


def compile_translit_tables(tables):
    maps = {}

    for name, table in tables.items():
        upper_and_lower = {
            src.upper(): dst.upper()
            for src, dst in table.items()
        }

        upper_and_lower.update(table)

        maps[name] = _compile_char_map(upper_and_lower)

    return maps


_translit_maps = compile_translit_tables(TRANSLIT_TABLES)


def transliterate(text, table_name):
    """
    Транслитерация по таблицам

    >>> transliterate(u'абвгдеёжзийклмнопрстуфхчцшщъыьэюя', 'cyr-lat')
    u'abvgdeyozhzijklmnoprstufxchczshshh_d_i_tyeyuya'
    >>> transliterate(u'abcdefghijklmnopqrstuvwxyz', 'lat-cyr')
    u'\u0430\u0431\u0446\u0434\u0435\u0444\u0433\u0445\u0438\u0439\u043a\u043b\u043c\u043d\u043e\u043f\u044f\u0440\u0441\u0442\u0443\u0432\u0432\u0445\u044b\u0437'
    """

    char_map = _translit_maps[table_name]

    return char_map(text)


def nbsp(value):
    """Заменяет все пробелы на &nbsp;"""

    if not value:
        return ""

    return re.sub("\s", "&nbsp;", value)


def cut_direction(title):
    return title.replace('направление', 'напр.')


def parse_number_with_postfix(number):
    """
    Разбивает строки вида "123ABC" (число с постфиксом) в число 123 и строковую часть "ABC"

    Возвращает пару (число, постфикс)

    В случае отсутствия числа возвращает (None, постфикс)
    В случае отсутствия постфикса возвращает (число, None)
    """

    if not number:
        return None, None

    digits, letters = re.match(ur'(\d*)(.*)', number).groups()

    return int(digits) if digits else None, letters or None


NORM_CHAR_MAP = {
    u'ё': u'е',
    u'ї': u'i',
    u'\u0456': u'i',
    u'ç': u'c',
    u'ğ': u'g',
    u'ı': u'i',
    u'ö': u'o',
    u'ş': u's',
    u'ü': u'u',
    u'i\u0307': u'i',  # İ (LATIN CAPITAL LETTER I WITH DOT ABOVE) lowercased in chrome, yandex browser
}

SPEC_RE = re.compile(ur'[№" ]+', re.U)

_norm_char_map = _compile_char_map(NORM_CHAR_MAP)


def normalize(text):
    if not text:
        return u''

    lower = text.lower()
    remapped = _norm_char_map(lower)
    clean = SPEC_RE.sub(' ', remapped).strip()

    return clean


SIMILAR_CHARS = {
    'lat': u'CAEOPHKXTcxyoepak',
    'cyr': u'САЕОРНКХТсхуоерак',
}


def _compile_similar_char_map(dst_script):
    maps = SIMILAR_CHARS.copy()
    dst_map = maps.pop(dst_script)
    src_maps = maps.values()

    return _compile_char_map({
        src_char: dst_char
        for dst_char, src_chars in zip(dst_map, zip(*src_maps))
        for src_char in src_chars
    })


similar_chars_to_cyr = _compile_similar_char_map('cyr')
similar_chars_to_lat = _compile_similar_char_map('lat')


PUNTO_CHARS = {
    'en': u"""qwertyuiop{[}]asdfghjkl:;'"zxcvbnm<,>.`""",
    'ru': u"""йцукенгшщзххъъфывапролджжээячсмитьббююё""",
    'uk': u"""йцукенгшщзххїїфівапролджжєєячсмитьббююё""",
}

PUNTO_DIRECTIONS = [
    ('en', 'ru'),
    ('ru', 'en'),
    ('en', 'uk'),
    ('uk', 'en')
]

_punto_maps = [
    _compile_char_map(dict(zip(PUNTO_CHARS[src], PUNTO_CHARS[dst])))
    for src, dst in PUNTO_DIRECTIONS
]


def punto_variants(text):
    result = []
    seen = set([text])

    for char_map in _punto_maps:
        mapped = char_map(text)

        if mapped not in seen:
            result.append(mapped)
            seen.add(mapped)

    return result


def md5_hex(s):
    from types import UnicodeType
    from hashlib import md5

    if type(s) == UnicodeType:
        s = s.encode('utf8')

    return md5(s).hexdigest()


PARTS_RE = re.compile('\s[\u2014-]\s+', re.UNICODE)


def mdash(value):
    """Заменяет дефис на тире"""
    if not value:
        return ''

    parts = PARTS_RE.split(value)

    return ' — '.join(parts)


def nowrap(value):
    return '<span class="g-nowrap">%s</span>' % value


def mdash_nowrap(value, last=False):
    """Заменяет дефис на тире, обертывает nowrap"""
    if not value:
        return ''

    parts = PARTS_RE.split(value)

    if last:
        # обертываем только последнюю часть
        parts[-1] = nowrap(parts[-1])

    else:
        parts = map(nowrap, parts)

    return ' — '.join(parts)


GOOD_SPLITTER = re.compile(r'(?<=[^ ^]{3}) (?!.{,3}' + r'[ \)\,\.$])', re.U | re.I | re.S)


def mdash_wrappable(value):
    """
    Заменяет дефис на тире, обертывает nowrap, оставляя возможность переноса в допустимых местах
    """

    if not value:
        return ''

    parts = PARTS_RE.split(value)

    nowrap_parts = []

    for part in parts:
        subparts = GOOD_SPLITTER.split(part)

        nowrap_parts.append(' '.join(map(nowrap, subparts)))

    return ' — '.join(nowrap_parts)


def nice_quotes(value):
    """Заменяет кавычки-лапки на кавычки-ёлочки"""
    if not value:
        return u''

    parts = value.split('"')

    # Меняем только если в строке две кавычки
    if len(parts) == 3:
        return parts[0] + '«' + parts[1] + '»' + parts[2]

    return value


def title_with_preposition(preposition, title):
    """
    Получает название с предлогом, соединенные через неразрывный пробел.
    Если название - пустая строка или None, то вернется None.
    :param preposition: предлог
    :param title: название
    """
    return u''.join((preposition, NBSP, title)) if title else None
