# -*- coding: utf-8 -*-

import logging
import sys
from threading import Thread, Lock

from django.conf import settings
from django.db import connection
from travel.avia.library.python.common.utils import environment, tracer


class SimpleThread(Thread):
    u""" Вспомогательный поток для функции run_in_thread """
    # общий Lock на обновление входных параметров
    # предполагается, что функция будет использовать лок
    _lock = Lock()

    def __init__(self, function, args, kwargs, lock_for_update, request=None):
        self._result = None
        self.request = request
        self.function = function
        self.args = args
        self.kwargs = kwargs
        self.lock_for_update = lock_for_update
        self._exc_info = None
        super(SimpleThread, self).__init__()

    def run(self):
        log = logging.getLogger(__name__)
        environment.activate_request(self.request)
        try:
            if self.lock_for_update:
                self.kwargs.update({'lock': self._lock})
            try:
                self._result = self.function(*self.args, **self.kwargs)
            finally:
                # Закрываем коннекшн, если он создался для треда
                self.close_connection()
        except Exception:
            self._exc_info = sys.exc_info()
            log.exception(u'Проблема запуска потока')

    def get_result(self):
        if self._exc_info is not None:
            raise self._exc_info[0], self._exc_info[1], self._exc_info[2]

        return self._result

    def close_connection(self):
        connection.close()


class FakeThread(SimpleThread):
    def start(self):
        self.run()

    def join(self, *args, **kwargs):
        pass

    def close_connection(self):
        pass


def run_in_thread(function, args=tuple(), kwargs=dict(), lock_for_update=False):
    u""" Запускает функцию в потоке, возвращает объект потока.
    После завершения работы потока результаты работы функции можно получить с
    помощью метода потока get_result(),
    lock_for_update - параметр позволяющий вызвать функцию с
    дополнительным именованым параметром lock, lock создается один на программу.
    (Точнее на инстанс класса SimpleThread)
    Если в ходе выполнения потока возникли исключения,
    функция при вызове get_result() кидает Exception со стек трейсом.
    Так же стек трейс пишется в лог rasp.common
    """

    request = environment.get_request()

    thread = SimpleThread(function, args, kwargs, lock_for_update, request)
    if settings.DEBUG:
        # Не стартовать ничего в тредах (это съедает sql-запросы)
        thread = FakeThread(function, args, kwargs, lock_for_update, request)

    tracer.start_thread(thread)
    return thread
