# -*- coding: utf-8 -*-

import json
import logging
from functools import partial
from itertools import imap

from django.utils import translation

from travel.avia.library.python.common.models.geo import Point, Station
from travel.avia.library.python.common.utils.exceptions import SimpleUnicodeException
from travel.avia.library.python.common.xgettext.i18n import gettext


log = logging.getLogger(__name__)


def to_json(obj):
    return unicode(json.dumps(obj, ensure_ascii=False, separators=(',', ':')))


class TitleGeneratorError(SimpleUnicodeException):
    pass


class PointNotFound(TitleGeneratorError):
    pass


class TitleThreadData(object):
    thread = None
    number = None
    supplier = None
    t_type = None
    rtstations = None
    has_virtual_end = None
    is_circular = None


class TitleGenerator(object):
    def __init__(self, thread, strategy=None):
        self.title = None
        self.title_short = None
        self.title_common = None
        self.title_rtstations = []

        self.strategy = strategy

        self.thread = thread
        self.data = TitleThreadData()

    def generate(self):
        thread_data = self.check_and_fill_data()

        strategy = self.get_strategy(thread_data)

        strategy.generate(self, thread_data)

    def get_strategy(self, thread_data):
        if self.strategy:
            return STRATEGY_CLASSES[self.strategy]()

        if thread_data.t_type and thread_data.t_type.code == 'suburban':
            strategy = 'suburban'
        elif thread_data.t_type and thread_data.t_type.code == 'urban':
            strategy = 'urban'
        elif thread_data.supplier and thread_data.supplier.code == 'mta':
            strategy = 'mta'
        else:
            strategy = 'default'

        return STRATEGY_CLASSES[strategy]()

    def check_and_fill_data(self):
        thread = self.thread

        data = self.data

        data.thread = thread
        data.number = thread.number
        data.supplier = self.get_supplier_from_thread(thread)
        data.t_type = thread.t_type_id and thread.t_type

        data.rtstations = getattr(thread, 'rtstations', None)

        if not data.rtstations and thread.id is not None:
            data.rtstations = list(thread.rtstation_set.all().select_related('station__settlement'))

        if not data.rtstations:
            raise TitleGeneratorError(u"Не указаны станции нитки, невозможно сгенерировать название")

        if len(data.rtstations) < 2:
            raise TitleGeneratorError(u"Не достаточная длинна маршрута")

        data.has_virtual_end = any((rts.is_virtual_end for rts in data.rtstations[1:]))
        data.is_circular = self.thread.is_circular or data.rtstations[0].station == data.rtstations[-1].station

        return data

    def get_supplier_from_thread(self, thread):
        if thread.supplier_id:
            return thread.supplier

        # TODO: Удалить код с route когда в нитке поставщик будет заполнен
        if hasattr(thread, 'route'):
            return thread.route.supplier_id and thread.route.supplier

    @classmethod
    def extract_point_keys(cls, title_dict):
        title_parts = title_dict['title_parts']
        keys = []
        for part in title_parts:
            if isinstance(part, basestring):
                keys.append(part)

                continue

            elif isinstance(part, dict):
                if part['type'] == 'mta_station_with_city':
                    keys.append(part['station'])
                    keys.append(part['settlement'])

                    continue

            raise TitleGeneratorError(u'Неожиданный тип части названия {}'.format(part))

        return keys

    @classmethod
    def L_title(cls, title_dict,
                prefetched_points=None, short=False, popular=False, lang=None):
        if lang is None:
            lang = translation.get_language()

        strategy_class = STRATEGY_CLASSES.get(title_dict['type'], STRATEGY_CLASSES['default'])
        if popular:
            return strategy_class.gen_popular_title(title_dict, lang, prefetched_points)
        elif short:
            return strategy_class.gen_short_title(title_dict, lang, prefetched_points)
        else:
            return strategy_class.gen_title(title_dict, lang, prefetched_points)


class TitleGenerationStrategy(object):
    def __init__(self):
        self._all_stations_inside_one_city = False
        self.title_rtstations = None
        self.prefetched_points = {}

    def get_combined_rtstations(self, data):
        combined_rtstations = []

        for rts in data.rtstations[1:-1]:
            if rts.is_combined:
                combined_rtstations.append(rts)

        return combined_rtstations

    def get_rtstation_virtual_end(self, data):
        if data.has_virtual_end:
            return imap(lambda rts: rts.is_virtual_end, data.rtstations[1:]).next()

    def lower_yo2ye_equal(self, str1, str2):
        return str1.lower().replace(u"ё", u"е") == str2.lower().replace(u"ё", u"е")

    def try_to_generalize(self, station):
        return generalize_point(station, self._all_stations_inside_one_city)


class DefaultTitleGenerationStrategy(TitleGenerationStrategy):
    def generate(self, generator, thread_data):
        self.title_rtstations = self.get_title_rtstations(thread_data)

        path_stations = [rts.station for rts in self.title_rtstations]

        self._all_stations_inside_one_city = \
            len(set([p.settlement_id for p in path_stations])) == 1

        title_dict = self.build_title_dict(thread_data, path_stations)

        if thread_data.thread.is_combined:
            title_dict['is_combined'] = True
            title_dict['t_type'] = thread_data.t_type and thread_data.t_type.code

        self.fill_generator_attributes(title_dict, generator)

    def fill_generator_attributes(self, title_dict, generator):
        generator.title = self.gen_title(title_dict, 'ru', self.prefetched_points)
        generator.title_short = self.gen_short_title(title_dict, 'ru', self.prefetched_points)

        generator.title_common = to_json(title_dict)
        generator.title_rtstations = self.title_rtstations

    def get_title_rtstations(self, data):
        # начальная станция
        title_rtstations = [data.rtstations[0]]

        title_rtstations += self.get_combined_rtstations(data)

        # первая виртуальная конечная
        virtual_end = self.get_rtstation_virtual_end(data)
        if virtual_end:
            title_rtstations.append(virtual_end)

        # последняя станция маршрута
        last_station = data.rtstations[-1]

        # если нитка колцевая, берем начальную и предпоследнюю,
        # но только в том случае, если маршрут без виртуальных конечных остановок
        if data.is_circular and \
           len(data.rtstations) > 2 and \
           not data.has_virtual_end:
            last_station = data.rtstations[-2]

        if last_station.station != (virtual_end and virtual_end.station):
            title_rtstations.append(last_station)

        return title_rtstations

    def build_title_dict(self, data, title_stations):
        title_parts = []

        for station in title_stations:
            point = self.try_to_generalize(station)

            title_parts.append(point.point_key)
            self.prefetched_points[point.point_key] = point

        return {
            'type': 'default',
            'title_parts': title_parts
        }

    @classmethod
    def get_point_title(cls, point, lang):
        return point.L_title(lang=lang) or u''

    @classmethod
    def get_point_short_title(cls, point, lang):
        """ Названия одинаковы по умолчанию """
        return point.L_title(lang=lang) or u''

    @classmethod
    def get_point_popular_title(cls, point, lang):
        return point.L_popular_title(lang=lang) or u''

    @classmethod
    def gen_title(cls, title_dict, lang, prefetched_points=None):
        def title_getter(point):
            return cls.get_point_title(point, lang)

        return cls._get_title(title_dict, lang, title_getter, prefetched_points)

    @classmethod
    def gen_short_title(cls, title_dict, lang, prefetched_points=None):
        def title_getter(point):
            return cls.get_point_short_title(point, lang)

        return cls._get_title(title_dict, lang, title_getter, prefetched_points)

    @classmethod
    def gen_popular_title(cls, title_dict, lang, prefetched_points=None):
        def title_getter(point):
            return cls.get_point_popular_title(point, lang)

        return cls._get_title(title_dict, lang, title_getter, prefetched_points)

    @classmethod
    def part2text(cls, part, title_getter, prefetched_points):
        if isinstance(part, basestring):
            try:
                point = prefetched_points[part]
            except KeyError as e:
                raise PointNotFound(e[0])

            return title_getter(point)

        elif isinstance(part, dict):
            if part['type'] == 'mta_station_with_city':
                try:
                    station = prefetched_points[part['station']]
                    settlement = prefetched_points[part['settlement']]
                except KeyError as e:
                    raise PointNotFound(e[0])

                # Пока эту фразу не загоняем в переводы, т.к. нужна она только для mta
                return u'{settlement} ({station})'.format(
                    station=title_getter(station),
                    settlement=title_getter(settlement)
                )

        raise TitleGeneratorError(u'Неожиданный тип части названия {} {}'.format(part, type(part)))

    @classmethod
    def _get_title(cls, title_dict, lang, title_getter, prefetched_points):
        if not prefetched_points:
            prefetched_points = Point.in_bulk(TitleGenerator.extract_point_keys(title_dict))

        route_title = u' - '.join([cls.part2text(part, title_getter, prefetched_points)
                                   for part in title_dict['title_parts']])

        if title_dict.get('add_circular'):
            route_title = gettext(u'{route_title} (кольцевой)', lang=lang).format(
                route_title=route_title)

        if title_dict.get('add_circular_mta'):
            route_title = gettext(u'{route_title} / кольцевой', lang=lang).format(
                route_title=route_title)

        if title_dict.get('add_ring'):
            route_title = gettext(u'{route_title} (кольцо)', lang=lang).format(
                route_title=route_title)

        if title_dict.get('is_combined'):
            if title_dict['type'] == 'suburban':
                # Фраза вида "Екатеринбург-Пасс. — Ощепково — Тюмень (согласованные поезда)"
                return gettext(u'{route_title} (согласованные поезда)', lang=lang
                               ).format(route_title=route_title)

            elif title_dict['t_type'] == 'train':
                if len(title_dict['title_parts']) == 3:
                    return gettext(u'{route_title} (с пересадкой)', lang=lang
                                   ).format(route_title=route_title)
                else:
                    return gettext(u'{route_title} (с пересадками)', lang=lang
                                   ).format(route_title=route_title)
            else:
                return route_title

        return route_title


class SuburbanTitleGenerationStrategy(DefaultTitleGenerationStrategy):
    def get_title_rtstations(self, data):
        # начальная станция
        title_rtstations = [data.rtstations[0]]

        title_rtstations += self.get_combined_rtstations(data)

        # первая виртуальная конечная
        virtual_end = self.get_rtstation_virtual_end(data)
        if virtual_end:
            title_rtstations.append(virtual_end)

        # последняя станция маршрута
        last_station = data.rtstations[-1]

        # если это электричка и нитка кольцевая
        if data.is_circular:
            # добавляем конечную станцию к точкам, только если есть вирт.конечная
            if data.has_virtual_end and last_station.station != (virtual_end and virtual_end.station):
                title_rtstations.append(last_station)

        elif last_station.station != (virtual_end and virtual_end.station):
            title_rtstations.append(last_station)

        return title_rtstations

    @classmethod
    def get_point_title(cls, point, lang):
        title = super(SuburbanTitleGenerationStrategy, cls).get_point_title(point, lang)

        if isinstance(point, Station) and point.t_type_id == 2:
            return gettext(u'аэропорт {title}', lang=lang).format(
                title=title
            )

        else:
            return title

    @classmethod
    def get_point_short_title(cls, point, lang):
        short_title = point.L_short_title(lang=lang) or u''

        if isinstance(point, Station) and point.t_type_id == 2:
            return gettext(u'а/п {short_title}', lang=lang).format(
                short_title=short_title
            )

        else:
            return short_title

    @classmethod
    def get_point_popular_title(cls, point, lang):
        popular_title = point.L_popular_title(lang=lang) or u''

        if isinstance(point, Station) and point.t_type_id == 2:
            return gettext(u'а/п {short_title}', lang=lang).format(
                short_title=popular_title
            )

        else:
            return popular_title

    def build_title_dict(self, data, title_stations):
        title_dict = {
            'type': 'suburban',
        }
        # результирующие массивы
        title_parts = []

        for station in title_stations:
            title_parts.append(station.point_key)
            self.prefetched_points[station.point_key] = station

        # если электричка, нитка кольцевая, то добавляем "кольцо"
        if data.is_circular and len(title_stations) == 1:
            title_dict['add_ring'] = True

        title_dict['title_parts'] = title_parts

        return title_dict


class UrbanTitleGenerationStrategy(DefaultTitleGenerationStrategy):
    def get_title_rtstations(self, data):
        # начальная станция
        title_rtstations = [data.rtstations[0]]

        # первая виртуальная конечная
        virtual_end = self.get_rtstation_virtual_end(data)
        if virtual_end:
            title_rtstations.append(virtual_end)

        # последняя станция маршрута
        last_rtstation = data.rtstations[-1]

        if last_rtstation.station != (virtual_end and virtual_end.station):
            title_rtstations.append(last_rtstation)

        return title_rtstations

    def build_title_dict(self, data, title_stations):
        title_dict = {
            'type': 'urban'
        }
        title_parts = []

        for station in title_stations:
            point = self.try_to_generalize(station)

            title_parts.append(point.point_key)
            self.prefetched_points[point.point_key] = point

        if data.is_circular and not data.has_virtual_end:
            title_dict['add_circular'] = True

        title_dict['title_parts'] = title_parts

        return title_dict


class MtaTitleGenerationStrategy(DefaultTitleGenerationStrategy):
    def get_title_rtstations(self, data):
        # начальная станция
        title_rtstations = [data.rtstations[0]]

        # первая виртуальная конечная
        virtual_end = self.get_rtstation_virtual_end(data)

        # последняя станция маршрута
        last_rtstation = data.rtstations[-1]

        # если нитка колцевая, и есть виртуальный конец формируем название 1 - 2 - 1
        if data.is_circular:
            if virtual_end and virtual_end.station != last_rtstation.station:
                title_rtstations.append(virtual_end)
                title_rtstations.append(last_rtstation)

        else:
            title_rtstations.append(last_rtstation)

        return title_rtstations

    def build_title_dict(self, data, title_stations):
        title_dict = {
            'type': 'mta'
        }
        title_parts = []

        def append_extreme_title_part(station):
            point = self.try_to_generalize(station)

            if isinstance(point, Station) and station.settlement_id:
                # Если нельзя обобщать, но все же есть город с другим названием
                # Москва (м. Обуховская)
                settlement = point.settlement
                if not self.lower_yo2ye_equal(station.settlement.L_title(lang='ru'),
                                              station.L_title(lang='ru')):
                    title_parts.append({
                        'type': 'mta_station_with_city',
                        'station': station.point_key,
                        'settlement': settlement.point_key
                    })

                    self.prefetched_points[settlement.point_key] = settlement
                    self.prefetched_points[station.point_key] = station

                # Названия совпали, без разницы что показывать, берем город
                else:
                    title_parts.append(settlement.point_key)
                    self.prefetched_points[settlement.point_key] = settlement

            else:
                title_parts.append(point.point_key)
                self.prefetched_points[point.point_key] = point

        def append_middle_title_part(station):
            point = station

            if station.settlement_id:
                point = station.settlement

            title_parts.append(point.point_key)
            self.prefetched_points[point.point_key] = point

        last_idx = len(title_stations) - 1

        for idx, station in enumerate(title_stations):
            # Если это средняя станция
            if idx == 0:
                append_extreme_title_part(station)
            elif idx == last_idx:
                append_extreme_title_part(station)
            else:
                append_middle_title_part(station)

        if data.is_circular and len(title_stations) == 1:
            title_dict['add_circular_mta'] = True

        title_dict['title_parts'] = title_parts

        return title_dict


STRATEGY_CLASSES = {
    'default': DefaultTitleGenerationStrategy,
    'suburban': SuburbanTitleGenerationStrategy,
    'urban': UrbanTitleGenerationStrategy,
    'mta': MtaTitleGenerationStrategy
}


def generalize_point(point, intracity_path=False):
    if not isinstance(point, Station):
        return point
    if intracity_path or point.not_generalize:
        return point
    if point.settlement_id:
        return point.settlement
    else:
        return point


def build_simple_title_common(t_type, points):
    return to_json({
        'type': 'default',
        'title_parts': [p.point_key for p in points],
        't_type': t_type.code
    })


def build_default_title_common(t_type, points, intracity_path=False):
    points = map(partial(generalize_point, intracity_path=intracity_path), points)

    return to_json({
        'type': 'default',
        'title_parts': [p.point_key for p in points],
        't_type': t_type.code
    })
