from __future__ import absolute_import

import codecs
import json
from functools import wraps
from operator import methodcaller
from os import path, getpid
from thread import get_ident
from time import time, strftime

from django.conf import settings


stacks = {}


class Frame(object):
    def __init__(self, name, attrs):
        self.name = name
        self.children = []
        self.attrs = attrs
        self.pid = None

    def enter(self):
        self.pid = getpid()

        enter = self.attrs['enter'] = {}

        enter['size'], enter['resident'] = get_size_resident(self.pid)
        enter['time'] = time()

    def leave(self):
        exit_ = self.attrs['exit'] = {'time': time()}

        if self.pid is not None:
            self.attrs['pid'] = self.pid
            exit_['size'], exit_['resident'] = get_size_resident(self.pid)

    def add(self, frame):
        self.children.append(frame)

    def set_attr(self, name, value):
        self.attrs[name] = value

    def __json__(self):
        o = {
            'name': self.name,
            'children': self.children,
        }

        o.update(self.attrs)

        return o

    def dump(self):
        log = path.join(settings.LOG_PATH, 'traces.log')

        f = codecs.open(log, 'a', encoding='utf-8')

        text = json.dumps(self, ensure_ascii=False, default=methodcaller('__json__'))

        t = self.attrs['exit']['time'] - self.attrs['enter']['time']

        print >>f, strftime('[%d/%b/%Y:%T %z]'), '%.2fms' % (t * 1000), self.attrs.get('url'), text

        f.close()


class Manager(object):
    def __init__(self, name, **attrs):
        self.name = name
        self.frame = Frame(name, attrs)

    def __enter__(self):
        try:
            if not getattr(settings, 'ENABLE_TRACER', False):
                return
        except ImportError:
            return

        stack = stacks.setdefault(get_ident(), [])

        if stack:
            stack[-1].add(self.frame)

        stack.append(self.frame)

        self.frame.enter()

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.frame.leave()

        ident = get_ident()

        try:
            stack = stacks[ident]
        except KeyError:
            return

        top_frame = stack.pop()

        while top_frame != self.frame:
            top_frame = stack.pop()

        if not stack:
            del stacks[ident]

        return False

    def leave(self):
        self.__exit__(None, None, None)


def _wrap(f, name):
    @wraps(f)
    def wrapper(*args, **kwargs):
        with Manager(name):
            return f(*args, **kwargs)

    return wrapper


def wrap(arg):
    if hasattr(arg, '__call__'):
        return _wrap(arg, arg.__name__)

    def decorator(f):
        return _wrap(f, arg)

    return decorator


def enter(name):
    manager = Manager(name)

    manager.__enter__()

    return manager


def start_thread(thread):
    manager = Manager('start thread')

    with manager:
        thread_manager = Manager('thread')

        thread.frame = thread_manager.frame

        thread_run = thread.run

        def wrapper():
            with thread_manager:
                thread.frame.set_attr('ident', get_ident())
                thread_run()

        thread.run = wrapper

        thread.start()

        manager.frame.add(thread.frame)


def join_thread(thread):
    manager = Manager('join thread', thread=thread.ident)

    with manager:
        thread.join()
        manager.frame.add(thread.frame)


def set_bottom_frame_attr(name, value):
    stack = stacks.get(get_ident())

    if stack:
        bottom_frame = stack[0]

        bottom_frame.set_attr(name, value)


def get_size_resident(pid):
    # size       total program size
    #            (same as VmSize in /proc/[pid]/status)
    # resident   resident set size
    #            (same as VmRSS in /proc/[pid]/status)
    # share      shared pages (from shared mappings)
    # text       text (code)
    # lib        library (unused in Linux 2.6)
    # data       data + stack
    # dt         dirty pages (unused in Linux 2.6)

    with open('/proc/%d/statm' % pid) as f:
        line = f.readline()

    fields = line.strip().split()

    assert len(fields) == 7

    size, resident, share, text, lid, data, dt = map(int, fields)

    return size, resident


def tag(tag):
    set_bottom_frame_attr('tag', tag)


def url(url):
    set_bottom_frame_attr('url', url)


def app(app):
    def wrapper(environ, start_response):
        manager = Manager('base')

        with manager:
            rv = app(environ, start_response)

        manager.frame.dump()

        return rv

    return wrapper
