# -*- coding: utf-8 -*-

import csv
import cStringIO
import codecs
from django.utils.itercompat import is_iterable


class UnicodeDictReader(csv.DictReader):
    def __init__(self, *args, **kwargs):
        self.encoding = kwargs.pop('encoding', 'utf8')
        self.errors = kwargs.pop('errors', 'strict')
        self.strip_values = kwargs.pop('strip_values', False)
        self.strip_keys = kwargs.pop('strip_keys', False)
        self.str_keys = kwargs.pop('str_keys', False)

        csv.DictReader.__init__(self, *args, **kwargs)

    @property
    def fieldnames(self):
        if self._fieldnames is None:
            row = self.reader.next()
            self._fieldnames = [self.transform_value(value) for value in row]

            if self._fieldnames[0].encode('utf8').startswith(codecs.BOM_UTF8):
                self._fieldnames[0] = self.fieldnames[0].encode('utf8')\
                    .replace(codecs.BOM_UTF8, '').decode('utf8')

            if self.str_keys:
                self._fieldnames = map(str, self.fieldnames)

            if self.strip_keys:
                self._fieldnames = map(lambda f: f.strip() if isinstance(f, basestring) else f, self.fieldnames)

        return self._fieldnames

    def next(self):
        rowdict = csv.DictReader.next(self)
        for key in rowdict:
            if not isinstance(rowdict[key], basestring) and is_iterable(rowdict[key]):
                rowdict[key] = [self.transform_value(value) for value in rowdict[key]]
            else:
                rowdict[key] = self.transform_value(rowdict[key])

        return rowdict

    def transform_value(self, value):
        if isinstance(value, basestring):
            return self.transform_text_value(value)
        else:
            return value

    def transform_text_value(self, value):
        new_value = value.decode(self.encoding, self.errors)

        if self.strip_values and new_value is not None:
            new_value = new_value.strip()

        return new_value


class UnicodeDictWriter(csv.DictWriter):
    def __init__(self, *args, **kwargs):
        self.encoding = kwargs.pop('encoding', 'utf8')
        self.errors = kwargs.pop('errors', 'strict')

        csv.DictWriter.__init__(self, *args, **kwargs)

    def _dict_to_list(self, rowdict):
        for k, value in rowdict.iteritems():
            rowdict[k] = self.convert_value(value)

        return csv.DictWriter._dict_to_list(self, rowdict)

    def convert_value(self, value):
        if isinstance(value, unicode):
            return value.encode(self.encoding, self.errors)
        elif value is None:
            return ""
        else:
            return unicode(value).encode(self.encoding, self.errors)

    def writehead(self):
        self.writerow(dict(zip(self.fieldnames, self.fieldnames)))

    def writerow_with_tail(self, rowdict, tail):
        return self.writer.writerow(self._dict_to_list(rowdict) + map(self.convert_value, tail))


class UnicodeReaderWrapper(object):
    def __init__(self, reader, encoding, errors, strip_values):
        self.reader = reader
        self.encoding = encoding
        self.errors = errors
        self.strip_values = strip_values

    def __iter__(self):
        return self

    def next(self):
        row = self.reader.next()
        return list(map(self.convert, row))

    def convert(self, value):
        if isinstance(value, str):
            if self.strip_values:
                value = value.strip()
            return value.decode(self.encoding, self.errors)
        else:
            return value


def unicode_reader(*args, **kwargs):
    encoding = kwargs.pop('encoding', 'utf8')
    errors = kwargs.pop('errors', 'strict')
    strip_values = kwargs.pop('strip_values', False)

    reader = csv.reader(*args, **kwargs)

    return UnicodeReaderWrapper(reader, encoding, errors, strip_values)


class UnicodeWriter:
    """
    A CSV writer which will write rows to CSV file "f",
    which is encoded in the given encoding.
    """

    def __init__(self, f, dialect=csv.excel, encoding="utf-8", **kwds):
        # Redirect output to a queue
        self.queue = cStringIO.StringIO()
        self.writer = csv.writer(self.queue, dialect=dialect, **kwds)
        self.stream = f
        self.encoder = codecs.getincrementalencoder(encoding)()

    def writerow(self, row):
        self.writer.writerow([s.encode("utf-8") for s in row])
        # Fetch UTF-8 output from the queue ...
        data = self.queue.getvalue()
        data = data.decode("utf-8")
        # ... and reencode it into the target encoding
        data = self.encoder.encode(data)
        # write to the target stream
        self.stream.write(data)
        # empty queue
        self.queue.truncate(0)

    def writerows(self, rows):
        for row in rows:
            self.writerow(row)
