# -*- coding: utf-8 -*-

import logging
import re
import socket
import urllib2
from datetime import date, timedelta
from xml.etree import ElementTree as ET
from xml.parsers.expat import ExpatError

from django.conf import settings
from django.utils.translation import get_language

from travel.avia.library.python.common.utils.caching import cached
from travel.avia.library.python.common.utils.http import quick_urlopen
from travel.avia.library.python.common.xgettext.i18n import gettext


log = logging.getLogger(__name__)


def weather_city(request, settlement):
    try:
        return get_weather_city_info(request, settlement._geo_id)

    except WeatherError:
        log.exception(u"Ошибка получения погоды города")
        return {}


def weather_data(request, station):
    """Погода на станции"""
    weather = {}

    try:
        if station.t_type_id == 1 and station.settlement_id:
            weather = weather_city(request, station.settlement)

        if station.t_type_id == 2:
            icao = station.get_code('icao') or station.get_code('weather_icao')
            if icao:
                weather = get_weather_airport_info(request, icao)

    except WeatherError:
        log.exception(u"Ошибка получения погоды станции")

    return weather


def translate_wind_direction(direction):
    table = dict(zip(u'сюзв', 'nswe'))
    return re.sub(ur'[сюзв]', lambda m: table[m.group(0)], direction.lower(), re.U)


def get_forecasts_url_by_geo_id(request, geo_id):
    """
    Возвращает url получения информации о погоде
    """

    return "http://%s/data/forecasts-by-geo/%s.xml" % (_get_weather_host(request), geo_id)


def get_weather_url(request, slug):
    """
    Возвращает url страницы информации о погоде в населенном пункте
    """

    return "http://%s/%s" % (_get_weather_host(request), slug)


def _get_weather_host(request):
    return settings.WEATHER_URL[request.tld] if request.tld in settings.WEATHER_URL else settings.WEATHER_DEFAULT_URL


class WeatherError(ValueError):
    pass


def get_strips_data(weather):
    strips = weather.find('strip_stat').findall('strip')

    strips_data = []

    states = {
        0: gettext(u'чисто и сухо'),
        1: gettext(u'влажно'),
        2: gettext(u'мокрая или местами вода'),
        3: gettext(u'иней или изморозь'),
        4: gettext(u'сухой снег'),
        5: gettext(u'мокрый снег'),
        6: gettext(u'слякоть'),
        7: gettext(u'лед'),
        8: gettext(u'уплотненный или укатанный снег'),
        9: gettext(u'замерзшая или неровная поверхность'),
        '-': None,
    }

    for strip in strips:
        strip_data = {}
        for p in strip:
            try:
                value = int(p.text)
            except ValueError:
                value = p.text

            if p.tag == 'rd':
                value = states[value]

            strip_data[p.tag] = value

        strips_data.append(strip_data)

    return strips_data


def weather_airport_info_key(request, icao):
    return '/weather/air/%s/%s' % (icao, get_language())


@cached(weather_airport_info_key)
def get_weather_airport_info(request, icao):
    """ Получение погоды по аэропорту
    Вход: icao код аэропорта
    Выход: словарь данных """
    if len(icao) != 4:
        raise WeatherError(u"Неверный код аэропорта")

    url = settings.WEATHER_ICAO_URL % icao

    try:
        info = quick_urlopen(url.encode('utf8', 'ignore')).read().strip()

        if info:
            tree = ET.fromstring(info)
            weather = tree.find('c').find('day').find('day_part')

            data = dict((p.tag, p.text) for p in weather)
            if data['wind_direction'] is None or len(data['wind_direction']) >= 4:
                data['wind_direction'] = ''

            try:
                data['strips'] = get_strips_data(weather)
            except (ExpatError, AttributeError, TypeError), e:
                # Данных по полосам нет, значит все в порядке
                data['strips'] = None
                data['strips_ok'] = True

            if 'humidity' in data and not re.match("^[\d+\.\,]$", data['humidity']):
                del data['humidity']

            data['url'] = get_weather_url(request, tree.find('c').attrib['link'].split("/")[-1])

            return data

        else:
            raise WeatherError(u"Не удалось получить данные с погоды (%s): пустой результат" % url)

    except (urllib2.HTTPError, urllib2.URLError, IOError, socket.error), e:
        raise WeatherError(u"Не удалось получить данные с погоды (%s): %s" % (url, e))
    except (ExpatError, AttributeError, TypeError), e:
        raise WeatherError(u"Ошибочный формат файла погоды (%s): %s" % (url, e))
    except Exception, e:
        raise WeatherError(u"Неизвестная ошибка при получении погоды(%s)" % url)


def weather_city_info_key(request, city_geo_id):
    return '/weather/geo/%s/%s' % (city_geo_id, get_language())


@cached(weather_city_info_key)
def get_weather_city_info(request, city_geo_id):
    """ Получение погоды по городу
    Вход: код города по погодной базе
    Выход: словарь данных """
    if not city_geo_id:
        return None

    def signed_number(number):
        if number > 0:
            return "+" + str(number)
        else:
            return str(number)

    url = get_forecasts_url_by_geo_id(request, city_geo_id)

    try:
        info = re.sub('xmlns[^\s]+', '', quick_urlopen(url).read().strip())

        if info:
            tree = ET.fromstring(info)
            weather = tree.find('fact')
            result = dict((p.tag, p.text) for p in weather)
            result['temperature'] = signed_number(int(result['temperature']))

            now = date.today().strftime("%Y-%m-%d")
            tomorrow = (date.today() + timedelta(days=1)).strftime("%Y-%m-%d")
            for day in tree.findall('day'):
                for day_part in day:
                    if day.get('date') == now and day_part.get('type') == "night_short":
                        result['tonight'] = signed_number(int(day_part.find('temperature').text))

                    if day.get('date') == tomorrow and day_part.get('type') == "day_short":
                        result['tomorrow'] = signed_number(int(day_part.find('temperature').text))

            result['url'] = get_weather_url(request, tree.attrib['slug'])
            result['city'] = tree.attrib['city']
            result['geo_id'] = city_geo_id
            if 'humidity' in result and not re.match("^[\d\.\,]+$", result['humidity']):
                del result['humidity']
            return result

        else:
            raise WeatherError(u"Не удалось получить данные с погоды (%s): пустой результат" % url)

    except (urllib2.HTTPError, urllib2.URLError, IOError, socket.error), e:
        raise WeatherError(u"Не удалось получить данные с погоды (%s): %s" % (url, e))
    except (ExpatError, AttributeError, TypeError), e:
        raise WeatherError(u"Ошибочный формат файла погоды (%s): %s" % (url, e))
    except Exception, e:
        raise WeatherError(u"Неизвестная ошибка при получении погоды(%s)" % url)
