# -*- coding: utf-8 -*-

"""
Утилиты использования библиотек яндекса
"""

import json
import logging
import socket
import time
import urllib
from datetime import datetime, timedelta
from hashlib import md5
from urllib2 import URLError

import cookiemy
from django.conf import settings
from django.utils.http import urlencode

from travel.avia.library.python.common.utils.http import quick_urlopen


log = logging.getLogger(__name__)

# http://wiki.yandex-team.ru/MyCookie/%D0%9D%D0%BE%D0%BC%D0%B5%D1%80%D0%91%D0%BB%D0%BE%D0%BA%D0%B0
MY_LANG = 39


def my_cookie(request):
    """
    Возвращает объект управления cookie my

    см. http://wiki.yandex-team.ru/MyCookie
    см. http://wiki.yandex-team.ru/IljaGolubcov/libcookiemypython
    """

    if 'my' not in request.COOKIES:
        return None

    cookie = urllib.unquote(request.COOKIES['my'])

    setup = cookiemy.Setup()

    try:
        setup.parse(cookie)
    except RuntimeError:  # RuntimeError: Wrong cookie version при некорректной куки
        return None

    return setup


def lang_detect_find(req):
    url = settings.LANGDETECTSERVER_URL
    json_req = json.dumps(req)

    try:
        json_res = quick_urlopen(url, data=json_req).read()
    except (URLError, socket.error):
        log.warning('langdetect read failed', exc_info=True)

        return None

    try:
        res = json.loads(json_res)
    except ValueError:
        log.warning('langdetect json parse failed (%r)' % json_res, exc_info=True)

        return None

    return res


def detect_lang(request, supported_langs=None):
    """
    Определяет язык страницы на основании настроек пользователя

    default_lang язык страницы по умолчанию
    supported_langs список возможных языков
    """

    supported_langs = supported_langs or request.supported_langs
    my = my_cookie(request)

    if my:
        my_lang_code = my.find(MY_LANG)

        my_lang = str(my_lang_code[-1]) if my_lang_code else 0

    else:
        my_lang = 0

    req = {
        'language': request.META.get('HTTP_ACCEPT_LANGUAGE', ''),
        'cookie': my_lang,
        'domain': request.host,
        'filter': ','.join(supported_langs),
        'pass-language': request.yauser.lang,
        'default': request.domain_lang
    }

    res = lang_detect_find(req)

    if not res:
        return request.domain_lang

    lang = res['id']

    if lang not in supported_langs:
        lang = request.domain_lang

    return lang


def get_country_by_lang(lang):
    """
    Возвращает название страны по языку

    Пример:
    'Ua' == get_country_by_lang('uk')
    'Ru' == get_country_by_lang('ru')
    """

    ret = lang_detect_find({'default': lang})

    if ret:
        return ret['name']

    return lang.capitalize()


def make_secret_key(request):
    """
    Формирует секретный ключ

    см. http://wiki.yandex-team.ru/AlekseyPonomarev/secret-key
    """

    days = int(time.time()) / 86400

    if request.yauser.is_authenticated():
        return "u%s" % md5('%s::%s' % (request.yauser.uid, days)).hexdigest()

    elif 'yandexuid' in request.COOKIES:
        return "y%s" % md5('0:%s:%s' % (request.COOKIES['yandexuid'], days)).hexdigest()

    return None


def changelang_url(request, lang):
    return "https://%s/api/lang/v1.1/save.xml?%s" % (request.tune_host, urlencode({
        'intl': lang,
        'retpath': request.build_absolute_uri(),
        'sk': request.secret_key
    }))


def has_valid_yandexuid(request):
    """ https://wiki.yandex-team.ru/Cookies/yandexuid """

    if settings.HAS_VALID_YANDEXUID_ALWAYS_TRUE:
        return True

    yandexuid_cookie_lifetime = 11

    yandexuid = request.COOKIES.get('yandexuid')

    try:
        dt = datetime.fromtimestamp(float(yandexuid[-10:]))

    except Exception:
        is_valid_yandexuid = False

    else:
        now = request.naive_msk_now
        past = now.replace(year=now.year - yandexuid_cookie_lifetime)
        is_valid_yandexuid = past < dt < now + timedelta(1)

    if not is_valid_yandexuid:
        data = {
            'HTTP_HOST': request.META.get('HTTP_HOST'),
            'REQUEST_URI': request.META.get('REQUEST_URI'),
            'HTTP_COOKIE': request.META.get('HTTP_COOKIE'),
            'REMOTE_ADDR': request.META.get('REMOTE_ADDR'),
            'HTTP_USER_AGENT': request.META.get('HTTP_USER_AGENT'),
        }

        log.info(json.dumps(data))

    return is_valid_yandexuid
