# -*- coding: utf-8 -*-

"""Система подбора тизеров"""

import random
from datetime import date

from django.db.models import Q
from django.db import connection

from travel.avia.library.python.common.models.geo import Settlement, Station, Country
from travel.avia.library.python.common.models.teasers import Teaser


class TeaserSetBase(object):
    """ Базовый класс, содержит методы для фильтрации тизеров
        требует переопределения методов get_teasers и render_teaser в проекте
    """

    TYPES = [m[0] for m in Teaser.MODES]

    def __init__(self, request, project, page_code, data=None):
        """ page_code - код страницы в www.models.Page
            data - вспомогательные данные, разные на разных страницах, например экземпляр Settlement или Station
        """
        self.request = request
        self.page_code = page_code
        if not isinstance(self.page_code, list):
            self.page_code = [page_code]
        self.data = data
        self.project = project

        self.selected = self.select_teasers()

        self.set_decoration()

    def __json__(self):
        return self.selected

    def select_teasers(self):
        selected = {}

        teasers_by_type = self.fetch_teasers()

        for type_ in self.TYPES:
            if teasers_by_type[type_]:
                teaser = random.choice(teasers_by_type[type_])
                selected[type_] = teaser

        return selected

    def get_teasers_by_type_with_selected(self):
        teasers_by_type = self.fetch_teasers()

        for type_ in self.TYPES:
            if teasers_by_type[type_]:
                teaser = random.choice(teasers_by_type[type_])
                teaser.selected = True

        return teasers_by_type

    # Различные фильтрации
    def add_company(self, query, company):
        return query | Q(companies=company)

    def add_settlement(self, query, settlement):
        return query | Q(settlements=settlement) | Q(countries=settlement.country)

    def add_station(self, query, station):
        query |= Q(stations=station) | Q(countries=station.country)
        external_directions = [
            marker.external_direction
            for marker in station.externaldirectionmarker_set
                                 .all().select_related('external_direction')
        ]
        if external_directions:
            query |= Q(directions__in=external_directions)

        return query

    def add_threads(self, query, data):
        # для рейсов пополнения нет ниток
        threads = set(segment.thread for segment in data if getattr(segment, 'thread', None))

        return query | Q(threads__in=threads)

    def add_schedule_routes(self, query, schedule_routes):
        threads = set(sr.thread for sr in schedule_routes if getattr(sr, 'thread', None))

        return query | Q(threads__in=threads)

    def add_search_segments(self, query, segments):
        threads = set(s.thread for s in segments if getattr(s, 'thread', None))

        return query | Q(threads__in=threads)

    def add_points(self, query, data):
        for point in data:
            # RASP-9467
            if isinstance(point, Station):
                query |= Q(stations=point)

            if isinstance(point, Settlement):
                query |= Q(settlements=point)

            if isinstance(point, Station) or isinstance(point, Settlement):
                query |= Q(countries=point.country)

            elif isinstance(point, Country):
                query |= Q(countries=point)

        return query

    def add_ext_directions(self, query, from_to_points):
        point_from, point_to = from_to_points
        if isinstance(point_from, Station) and isinstance(point_to, Station):
            dirs1 = [marker.external_direction for marker in
                     point_from.externaldirectionmarker_set.all().select_related(
                         'external_direction')]
            dirs2 = [marker.external_direction for marker in
                     point_to.externaldirectionmarker_set.all().select_related(
                         'external_direction')]

            if len(dirs2) < len(dirs1):
                dirs1, dirs2 = dirs2, dirs1

            external_directions = [d for d in dirs1 if d in dirs2]

            return query | Q(directions__in=external_directions)

        return query

    def add_stations_pairs(self, query, segments):
        from_to_stations_pairs = list(set(
            (segment.station_from.id, segment.station_to.id) for segment in segments if
            not getattr(segment, 'is_transfer_segment', False)
        ))

        sql = '''
            SELECT external_direction_id
            FROM www_externaldirectionmarker
            WHERE station_id in (%d, %d)
            GROUP BY external_direction_id
            HAVING count(station_id) >= 2'''

        if from_to_stations_pairs:
            res_sql = sql % from_to_stations_pairs[0]

            for station_pair in from_to_stations_pairs[1:]:
                res_sql += ' UNION ' + sql % station_pair

            cursor = connection.cursor()
            cursor.execute(res_sql)

            ids = [row[0] for row in cursor.fetchall()]

            if ids:
                query |= Q(directions__id__in=ids)

        return query

    def filter_max_importance(self, teasers):
        # RASP-4569, отбор по важности
        if not teasers:
            return []
        max_importance = max((t.importance for t in teasers))
        return [t for t in teasers if t.importance == max_importance]

    def filter_hard_params(self, query):
        today = date.today()

        hard_params = {
            'is_active_' + self.project: True,
            'date_start__lte': today,
            'date_finish__gte': today
        }
        return query & Q(**hard_params)

    def get_teasers_by_type(self, teasers, filter_by_max_importance=True):
        teasers_by_type = dict((t, []) for t in self.TYPES)
        for t in teasers:
            teasers_by_type[t.mode].append(t)

        if filter_by_max_importance:
            for type_ in self.TYPES:
                teasers_by_type[type_] = self.filter_max_importance(teasers_by_type[type_])

        return teasers_by_type

    def fetch_teasers(self, filter_by_max_importance=True):
        teasers = []

        for page in self.page_code:
            teasers += self.get_teasers(page)

        return self.get_teasers_by_type(teasers, filter_by_max_importance=filter_by_max_importance)

    def set_decoration(self):
        u"""На главной странице надо отмечать, какой тизер самый верхний"""
        if (
            self.selected.get('special')
            and (not self.selected.get('ahtung') or self.project == 'ticket')
        ):
            self.selected['special'].bold_mode = True

    def get_teasers(self, page):
        """ Достает тизеры с учетом страницы """
        raise NotImplementedError

    def render_teaser(self, teaser_code, base=None):
        """ Шаблонизация """
        raise NotImplementedError


class TeaserSetRaspBase(TeaserSetBase):
    def __init__(self, project, page_code, data=None, national_version=None, language=None):
        self.national_version = national_version
        self.language = language
        super(TeaserSetRaspBase, self).__init__(None, project, page_code, data)

    def get_teasers(self, page):
        if page not in self.page_code:
            return []

        # Собираем запрос, который будет извлекать подходящие баннеры
        query = Q(pages__code='all') | Q(pages__code=page)

        if self.data:
            if page == 'company':
                company = self.data

                query = self.add_company(query, company)

            elif page == 'info_settlement':
                settlement = self.data

                query = self.add_settlement(query, settlement)

            elif page == 'info_station':
                station = self.data

                query = self.add_station(query, station)

            elif page == 'tablo':
                station, schedule_routes = self.data

                query = self.add_station(query, station)
                query = self.add_schedule_routes(query, schedule_routes)

            elif page == 'search_suburban':
                from_to_points = self.data['points']
                segments = self.data['routes']

                for point in from_to_points:
                    if isinstance(point, Settlement):
                        query |= Q(settlements=point)

                query = self.add_ext_directions(query, from_to_points)
                query = self.add_search_segments(query, segments)
                query = self.add_stations_pairs(query, segments)

            elif page == 'search':
                from_to_points = self.data['points']
                segments = self.data['routes']

                query = self.add_points(query, from_to_points)
                query = self.add_ext_directions(query, from_to_points)

                if segments:
                    companies = set(segment.company for segment in segments)
                    query |= Q(companies__in=companies)

                    query = self.add_search_segments(query, segments)

                    # Если любая пара станций от-до из результатов поиска есть среди маркеров направления, то берем тизеры этого направления
                    query = self.add_stations_pairs(query, segments)

            elif page.startswith('thread_'):
                thread, company = self.data

                query |= Q(threads=thread)
                if company:
                    query |= Q(companies=company)

            elif page == 'direction':
                ext_direction = self.data

                query |= Q(directions=ext_direction)

                countries = set([s.country_id for s in ext_direction.stations if s.country_id])

                query |= Q(countries__in=countries)

        teaser_qs = Teaser.get_objects(self.national_version, self.language)

        query = self.filter_hard_params(query)
        return list(teaser_qs.filter(query).distinct())

    def render_teaser(self, teaser_code, base=None):
        return ''


class TeaserSetRasp(TeaserSetRaspBase):
    def __init__(self, request, page_code, data=None, national_version=None, language=None):
        """
        :param request: Deprecated, not used.
        """
        super(TeaserSetRasp, self).__init__('rasp', page_code, data, national_version, language)


class TeaserSetMobile(TeaserSetRasp):
    def fetch_teasers(self):
        teasers = []

        for page in self.page_code:
            teasers += self.get_teasers(page)

        teasers = [t for t in teasers if t.mobile_content and t.mode == 'ahtung']

        return self.get_teasers_by_type(teasers)


def fetch_rasp_teasers(page_codes, data=None, filter_by_max_importance=True, national_version=None, language=None):
    teaser_set_rasp = TeaserSetRasp(None, page_codes, data, national_version, language)
    return teaser_set_rasp.fetch_teasers(filter_by_max_importance)
