# -*- coding: utf-8 -*-

import pytz

from datetime import datetime, timedelta

from ..models.geo import Station, StationTerminal, StationMajority
from ..models.schedule import RTStation
from ..models_utils.i18n import RouteLTitle
from ..models.transport import TransportModel
from ..utils import environment
from ..utils.date import FuzzyDateTime, get_near_date_future, MSK_TZ, human_duration
from ..utils.exceptions import SimpleUnicodeException
from ..xgettext.i18n import xgettext


def make_stops(thread, rtstations, start_dt):
    station_ids = [rts.station_id for rts in rtstations]
    stations = Station.objects.select_related('station_type', 'settlement', 'majority').in_bulk(station_ids)

    terminals = StationTerminal.objects.in_bulk([rts.terminal_id for rts in rtstations])

    t_model_ids = [rts.departure_t_model_id for rts in rtstations]
    t_models = TransportModel.objects.in_bulk(t_model_ids)

    stops = []
    naive_start_dt = start_dt.replace(tzinfo=None)

    for rts in rtstations:
        station = rts.station = stations[rts.station_id]
        city = station.settlement
        terminal = terminals.get(rts.terminal_id)

        t_model_id = rts.departure_t_model_id

        if t_model_id in t_models:
            segment_t_model = t_models[t_model_id]

        else:
            segment_t_model = thread.t_model

        stop_td = timedelta(0)

        if rts.tz_arrival is not None and rts.tz_departure is not None:
            stop_td = timedelta(minutes=rts.tz_departure - rts.tz_arrival)

        if rts.tz_arrival is None:
            arrival_dt = None

        else:
            arrival_dt = pytz.timezone(rts.time_zone).localize(
                naive_start_dt + timedelta(minutes=rts.tz_arrival)
            )

        if rts.tz_departure is None:
            departure_dt = None

        else:
            departure_dt = pytz.timezone(rts.time_zone).localize(
                naive_start_dt + timedelta(minutes=rts.tz_departure)
            )

        duration_td = arrival_dt and arrival_dt - start_dt

        local_arrival_dt = arrival_dt and arrival_dt.astimezone(station.pytz)
        local_departure_dt = departure_dt and departure_dt.astimezone(station.pytz)

        if rts.is_fuzzy:
            if local_arrival_dt and local_departure_dt:
                stop_time = local_departure_dt - local_arrival_dt

                local_departure_dt = FuzzyDateTime.fuzz(local_departure_dt)
                local_arrival_dt = FuzzyDateTime(local_departure_dt - stop_time)

            else:
                local_arrival_dt = local_arrival_dt and FuzzyDateTime.fuzz(local_arrival_dt)
                local_departure_dt = local_departure_dt and FuzzyDateTime.fuzz(local_departure_dt)

        stop_dict = {
            'title': station.L_title() or '',
            'id': station.id,
            'city': city,
            'station': station,
            'terminal': terminal,
            'departure_t_model': segment_t_model,
            'city_size': city and city.city_size() or '',
            'majority': station.majority_id or StationMajority.IN_TABLO_ID,
            'arrival': local_arrival_dt,
            'departure': local_departure_dt,
            'duration': duration_td,
            'stop_time': stop_td,
            'thread_number': thread.number,
            'majority_code': station.majority.code,
            'is_technical_stop': rts.is_technical_stop,
            'L_platform': rts.L_platform,
            'l_platform': rts.L_platform(),
            'is_fuzzy': rts.is_fuzzy,
            'is_combined': rts.is_combined,
        }

        stops.append(stop_dict)

    return stops


def get_thread_departure_date(requested_departure_date, thread):
    if (requested_departure_date and
            isinstance(requested_departure_date, datetime)):
        requested_departure_date = requested_departure_date.date()

    near_departure_date = get_near_date_future(thread.year_days)

    departure_date = requested_departure_date or near_departure_date
    if not departure_date:
        raise ThreadWithoutDaysError(u"rthread %s has no start date" % unicode(thread.id))

    return departure_date


class ThreadDataError(SimpleUnicodeException):
    pass


class ThreadWithoutStationsError(ThreadDataError):
    u""" Нить меньше чем с двумя станциями """
    pass


class ThreadWithoutDaysError(ThreadDataError):
    u""" Нить без дней хождения """
    pass


def get_interval_string(thread):
    if thread.density:
        return thread.density

    if not thread.period_int:
        return u''

    duration = timedelta(minutes=thread.period_int)

    return xgettext(u'интервал: <interval_duration />', interval_duration=human_duration(duration))


def get_thread_context(thread, departure_date):
    """ Получает данные о нитке
        Вход: нитка, дата отправления
        Выход: словарь
    """
    now_aware = environment.now_aware()

    query_type = 'thread'

    route = thread.route

    # RASP-3968
    if route.hidden:
        raise ThreadDataError(u"Route is hidden")

    if thread.type.code == 'cancel':
        raise ThreadDataError(u'Thread is cancelled')

    route_threads = list(route.rthread_set.all().select_related())

    t_model = thread.t_model

    tz_start_time = thread.tz_start_time
    start_date = departure_date

    RouteLTitle.fetch([obj.L_title for obj in [thread] + route_threads])

    thread_context = {
        'thread_number': thread.number,
        'thread_uid': thread.uid,
        'thread_id': thread.id,
        'thread': thread,
        'title': thread.title,
        'l-title': thread.L_title(),
        'transportType': thread.t_type_id,
        't_type': thread.t_type,
        'type': query_type,
        'route_threads': route_threads,
        't_model': t_model,
        'is_interval': thread.is_interval,
        'begin_time': thread.begin_time,
        'end_time': thread.end_time,
        'density': get_interval_string(thread),
        'comment': thread.comment,
    }

    rtstations = list(RTStation.objects.filter(thread__id=thread.id).order_by('id'))
    # thread_id, arrival, departure, day_shift, station_id, terminal_id, departure_t_model_id,
    # is_technical_stop, platform, is_fuzzy, is_combined

    start_dt = thread.pytz.localize(datetime.combine(start_date, tz_start_time))

    thread_context['start_datetime'] = start_dt

    thread_context['one_t_model'] = True

    stops = make_stops(thread, rtstations, start_dt)

    t_model = thread.t_model
    for stop in stops:
        if stop['departure_t_model'] != t_model:
            # Показываем по сегментам
            thread_context['one_t_model'] = False
            break

    current_station_id = None
    for stop in stops:
        if stop['arrival'] is not None and stop['arrival'] < now_aware:
            current_station_id = stop['station'].id

    # В итоге, если рейс уже ушел будет совпадать с последней стацией,
    # Возможно это неправильно, но так было до рефакторинга
    thread_context['currentStation'] = current_station_id

    if len(stops) < 2:
        raise ThreadWithoutStationsError(u"Слишком мало станций у нитки '%s'" % thread.id)

    stops[0]['arrival'] = None
    stops[-1]['departure'] = None

    thread_context['departure'] = start_dt.astimezone(MSK_TZ)
    thread_context['stops'] = stops

    # FIXME: для совместимости с мобильной версией, т.к. там жуткий код
    thread_context['stations'] = stops

    return thread_context
