# -*- coding: utf-8 -*-

from __future__ import absolute_import

import os.path
import json

from cachetools.func import lru_cache
from library.python import resource


KEYSETS_PATH = os.path.join(os.path.dirname(__file__), 'common_keysets')


@lru_cache(maxsize=None)
def get_common_keyset(name):
    return json.loads(_read_keyset(name))


def get_common_keyset_path(name, base_path=KEYSETS_PATH):
    return os.path.join(base_path, name + '-keyset.json')


def _read_keyset(name):
    base_path = os.path.join('resfs/file/', __package__.replace('.', '/'), 'common_keysets')
    path = get_common_keyset_path(name, base_path=base_path)
    return resource.find(path)


def xgettext_month(month, lang):
    """
    Возвращает локализованное название месяца по номеру:

    xgettext_month(1, 'ru') == 'Январь'
    ...
    xgettext_month(12, 'ru') == 'Декабрь'
    """

    return _get_month(month, lang, 'n%s')


def xgettext_month_genitive(month, lang):
    """
    Возвращает локализованное название месяца в родительном падеже по номеру:

    xgettext_month_genitive(1, 'ru') == 'Января'
    ...
    xgettext_month_genitive(12, 'ru') == 'Декабря'
    """

    return _get_month(month, lang, 'g%s')


def xgettext_month_short(month, lang):
    """
    Возвращает локализованное название месяца по номеру:

    xgettext_month_short(1, 'ru') == 'янв'
    ...
    xgettext_month_short(12, 'ru') == 'дек'
    """

    return _get_month(month, lang, 'short%s')


def xgettext_weekday(weekday, lang):
    """
    Возвращает локализованное название дня недели по номеру:

    xgettext_weekday(1, 'ru') == 'понедельник'
    ...
    xgettext_weekday(7, 'ru') == 'воскресенье'
    """

    return _get_weekday(weekday, lang, 'n%s')


def xgettext_weekday_short(weekday, lang):
    """
    Возвращает локализованное сокращенное название дня недели по номеру:

    xgettext_weekday(1, 'ru') == 'пн'
    ...
    xgettext_weekday(7, 'ru') == 'вc'
    """

    return _get_weekday(weekday, lang, 'short%s')


def xgettext_weekdays(lang, letter='n'):
    weekday_translations = get_common_keyset('days')[lang]['days']

    weekdays = []
    for i in range(1, 8):
        weekdays.append(weekday_translations['{}{}'.format(letter, i)])

    return weekdays


def get_datetemplate_translation(lang, key):
    keyset = get_common_keyset("datetemplates")
    template = None

    try:
        template = keyset[key]['translations'][lang]['form']

    except KeyError:
        pass

    return template if template else keyset[key]['translations']['ru']['form']


DAYS_KEYSET_DATA = get_common_keyset('days')
MONTHS_KEYSET_DATA = get_common_keyset('months')


def _get_month(month, lang, tmpl):
    return MONTHS_KEYSET_DATA[lang]['months'][tmpl % month]


def _get_weekday(weekday, lang, tmpl):
    return DAYS_KEYSET_DATA[lang]['days'][tmpl % weekday]


def get_schedule_autotext(key, lang):
    keyset = get_common_keyset("schedule_autotexts")
    template = None

    try:
        template = keyset[key]['translations'][lang]['form']

    except KeyError:
        pass

    return template if template else keyset[key]['translations']['ru']['form']
