from __future__ import unicode_literals

import logging
from time import time

from itertools import chain


class EventConverter(object):

    @staticmethod
    def get_str_level(int_level):
        return logging.getLevelName(int_level).lower()

    @staticmethod
    def convert(event):
        variables = []
        converted = {
            'additional': {
                'event_id': event['event_id'],
                'logger': event['logger'],
                'vars': variables,
            },
            'level': EventConverter.get_str_level(event['level']),
            'timestamp': int(round(time() * 1000)),

            'platform': event.get('platform'),
            'method': None,
            'host': event.get('server_name'),
            'file': None,
            'message': None,
            'line': None,
            'col': None,
        }

        request_data = event.get('request')
        if request_data:
            converted['additional']['request'] = request_data
            if 'url' in request_data:
                converted['url'] = request_data['url']

        exception_type = None
        exceptions = event.get('exception', {})
        if exceptions:
            stack = []
            for exception in exceptions['values']:
                trace = exception['stacktrace']
                module = exception['module'] or ''
                if module:
                    module = '%s.' % module

                exc_message = '%s%s: %s' % (module, exception['type'], exception['value'])
                exception_type = exception['type']

                for frame in trace['frames'] or []:
                    path = frame['abs_path']
                    func = frame['function']
                    lineno = frame['lineno']
                    stack.append(
                        '  File "%s", line %s, in %s\n    %s' % (
                            path,
                            lineno,
                            func,
                            (frame.get('context_line') or '').strip(),
                        )
                    )
                    converted.update({'file': path, 'method': func, 'line': lineno})

                    # Пробрасываем вычищенные локальные переменные.
                    variables.append({
                        'loc': '%s:%s %s()' % (path, lineno, func),
                        'vars': frame.get('vars', {}),
                    })

            if stack:
                converted['stack'] = '\n'.join(
                    chain(['Traceback (most recent call last):'], stack, [exc_message])  # noqa
                )

        sentry_message = event['sentry.interfaces.Message']
        if exception_type:
            converted['message'] = '[{}] [{}] {}'.format(event['logger'], exception_type, sentry_message['message'])
        else:
            converted['message'] = '[{}] {}'.format(event['logger'], sentry_message['message'])
        converted['additional']['message'] = sentry_message['formatted']

        return converted
