# -*- encoding: utf-8 -*-
import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from feature_flag_client.ab_content import IABFlagsContent  # noqa


class Context(object):
    def __init__(self, flags, ab_flags=None):
        self._flags = flags
        self._ab_flags = ab_flags or set()

    def flag_enabled(self, flag_code, ab_flags_content=None):
        # type: (str, IABFlagsContent) -> bool

        if self.use_ab(flag_code) and ab_flags_content:
            return ab_flags_content.flag_enabled(flag_code)
        return flag_code in self._flags

    def get_enabled_flags(self):
        return self._flags.copy()

    def use_ab(self, flag_code):
        return flag_code in self._ab_flags

    def __repr__(self):
        return u'<Context flags={!r}, abFlags={!r}>'.format(
            u','.join(self._flags),
            u','.join(self._ab_flags),
        )


class Client(object):
    def __init__(self, host, service_code, logger, timeout=0.1):
        self._host = host
        self._service_code = service_code
        self._logger = logger
        self._timeout = timeout

    def create_context(self):
        try:
            return self._create_context()
        except Exception as e:
            self._logger.exception('Can not create context %r', e)
        return Context(set())

    def _create_context(self):
        r = self._requests_retry_session().get(
            'http://{}/feature-flag?service-code={}'.format(
                self._host,
                self._service_code,
            ),
            timeout=self._timeout
        )
        r.raise_for_status()

        data = r.json()

        flags = set(data['flags'])
        ab_flags = set(data.get('abFlags') or [])
        return Context(flags, ab_flags)

    @staticmethod
    def _requests_retry_session(
            retries=3,
            status_forcelist=(500, 502, 504),
    ):
        session = requests.Session()
        retry = Retry(
            total=retries,
            read=retries,
            connect=retries,
            backoff_factor=0.01,  # 0ms 20ms 40ms
            status_forcelist=status_forcelist,
        )
        adapter = HTTPAdapter(max_retries=retry)
        session.mount('http://', adapter)
        return session
