# -*- encoding: utf-8 -*-
import logging
import time

from feature_flag_client import Client, Context


LOG = logging.getLogger(__name__)
MINUTE = 60


class FakeStorage(object):
    def __init__(self, *args, **kw):
        self._flags = set()

    def flag_enabled(self, flag_code):
        return flag_code in self._flags

    def append_flag(self, flag):
        self._flags.add(flag)

    def set_flags(self, *flags):
        self._flags = set(flags)


class Storage(object):
    def __init__(self, client, cache_ttl=5*MINUTE, logger=None):
        # type: (Client, int, logging.Logger) -> None
        self._client = client
        self._cache_ttl = cache_ttl
        self._end_time_of_cache = time.time()
        self._context = _get_empty_context()
        self._logger = logger or LOG

    @property
    def cache_ttl(self):
        return self._cache_ttl

    def reset_context(self):
        self._logger.info('Reset feature flag context')
        self._end_time_of_cache = time.time()

    def context(self):
        self._update_context_if_needed()
        return self._context

    def _update_context_if_needed(self):
        now = time.time()
        if now >= self._end_time_of_cache:
            self._logger.info('Updated feature flag context')
            self._context = self._client.create_context()
            self._end_time_of_cache = now + self._cache_ttl

    def flag_enabled(self, flag_code, ab_flags_content=None):
        return self.context().flag_enabled(flag_code, ab_flags_content)


def _get_empty_context():
    return Context([])


def get_storage(host, service_code, cache_ttl=5*MINUTE, logger=LOG, timeout=0.1):
    """
    Returned feature flag storage with implemented simple cache

    :param host: feature flag server host
    :param service_code: service code registered on the feature flag server
    :param cache_ttl: cache ttl in seconds
    :param timeout: timeout for request to feature flag server
    """
    return Storage(
        client=Client(
            host=host,
            service_code=service_code,
            logger=logger,
            timeout=timeout,
        ),
        cache_ttl=cache_ttl,
        logger=logger,
    )
