# -*- coding: utf-8 -*-
from __future__ import absolute_import, unicode_literals

import logging
from datetime import datetime

import pytz
from six.moves.urllib_parse import parse_qsl


MSK_TZ = pytz.timezone('Europe/Moscow')

logger = logging.getLogger(__name__)


class WsgiMiddleware(object):
    def __init__(self, app):
        self.app = app

    def __call__(self, environ, start_response):
        return self.app(environ, start_response)


class HTTPSchemeProxy(WsgiMiddleware):
    def __call__(self, environ, start_response):
        scheme = environ.get('HTTP_X_FORWARDED_PROTOCOL', '')

        if scheme:
            environ['wsgi.url_scheme'] = scheme

        return self.app(environ, start_response)


class Now(WsgiMiddleware):
    def __call__(self, environ, start_response):
        now = datetime.now()
        environ['now'] = environ['msk_now'] = MSK_TZ.localize(now)
        environ['naive_msk_now'] = environ['msk_now'].replace(tzinfo=None)

        return self.app(environ, start_response)


class ABExperimentFlags(WsgiMiddleware):
    def __call__(self, environ, start_response):
        flags = {}
        raw_flags = ''
        qs = environ.get('QUERY_STRING', '')

        try:
            query = dict(parse_qsl(qs))
            back_flags = query.get('back_flags')
            if back_flags:
                flags = dict(
                    flag.split('=', 1) for flag in back_flags.split(',')
                )
                raw_flags = back_flags
        except Exception as exc:
            logger.exception('ABExperimentFlags middleware %r', exc)
        environ['ab_experiment_flags'] = flags
        environ['ab_experiment_raw_flags'] = raw_flags

        return self.app(environ, start_response)
