# encoding: utf-8
from enum import Enum

from travel.avia.library.python.shared_dicts.common import iter_protobuf_data
from travel.library.python.dicts.base_repository import BaseListRepository
from travel.proto.shared_flights.flying_time.flying_time_pb2 import TFlyingTime


class FlyingTimesRepository(BaseListRepository):
    _PB = TFlyingTime


class ResourceType(Enum):
    FLYING_TIMES_SANDBOX_RESOURCE = 'AVIA_SHARED_FLIGHTS_FLYING_TIME_PRODUCTION_RESOURCE'


class FlyingTimesCache:
    def __init__(self, logger):
        self.flying_time_per_day = {}
        self.flying_time_min = {}
        self.logger = logger

    def get_flying_time(self, settlement1_id, settlement2_id, date_str):
        if not settlement1_id or not settlement2_id or settlement1_id == settlement2_id:
            return None
        if not date_str:
            return self.flying_time_min.get((settlement1_id, settlement2_id))
        time_by_date = self.flying_time_per_day.get((settlement1_id, settlement2_id, date_str))
        if time_by_date:
            return time_by_date
        flying_time = self.flying_time_min.get((settlement1_id, settlement2_id))
        if flying_time:
            return flying_time
        self.logger.warning(
            'No flying time for settlement_ids %d and %d in the cache',
            settlement1_id,
            settlement2_id,
        )

    def clear_cache(self):
        self.flying_time_per_day = {}
        self.flying_time_min = {}

    def populate(self):
        (
            self.flying_time_per_day,
            self.flying_time_min,
        ) = self.build_cache()
        self.logger.info(
            'populated cache with %d times per date and %d min times',
            len(self.flying_time_per_day),
            len(self.flying_time_min),
        )

    @staticmethod
    def build_cache():
        flying_time_per_day = {}
        flying_time_min = {}

        resource_type_to_repo = {
            ResourceType.FLYING_TIMES_SANDBOX_RESOURCE: FlyingTimesRepository,
        }

        for flying_time in iter_protobuf_data(resource_type_to_repo, ResourceType.FLYING_TIMES_SANDBOX_RESOURCE):
            if flying_time.FlightDate:
                key = (flying_time.DepartureSettlementId, flying_time.ArrivalSettlementId, flying_time.FlightDate)
                flying_time_per_day[key] = flying_time.MinFlyingTime
            else:
                key = (flying_time.DepartureSettlementId, flying_time.ArrivalSettlementId)
                flying_time_min[key] = flying_time.MinFlyingTime

        return (
            flying_time_per_day,
            flying_time_min,
        )
