# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import re

from django.contrib.contenttypes.models import ContentType
from django.db import models
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models.geo import Country, Settlement, Station, PointSynonym
from travel.avia.library.python.common.models_utils.i18n import L_field
from travel.avia.library.python.common.utils.caching import CachingManager
from travel.avia.library.python.common.utils.fields import TrimmedCharField
from travel.avia.library.python.common.utils.text import normalize


class SearchResult(object):
    def __init__(self, model, ids):
        self.model = model
        self.ids = set(ids)

    @cached_property
    def objects(self):
        objects = self.model._default_manager.in_bulk_list(self.ids)

        return objects

    def filtered(self, function):
        return filter(function, self.objects)


class NameSearchIndex(models.Model):
    SPLIT_RE = re.compile(r'[ \-—().]+', re.U)

    STOPWORDS = set(u'на с\о км st п c el km пос. тц'.split())

    SEARCH_FIELDS = {
        Country: {'L_title': ['nominative']},
        Settlement: {'L_title': ['nominative']},
        Station: {'L_title': ['nominative'], 'L_popular_title': ['nominative']}
    }

    class Meta:
        app_label = 'www'
        index_together = [
            ['text', 'content_type'],
            ['text', 'exact', 'content_type'],
        ]

    text = models.CharField(max_length=255, null=False, db_index=True)
    exact = models.BooleanField(null=False)

    content_type = models.ForeignKey(ContentType, db_index=True)
    object_id = models.PositiveIntegerField()

    _precache = None

    @classmethod
    def precache(cls):
        cls._precache = {}

        for model in cls.SEARCH_FIELDS:
            ct = ContentType.objects.get_for_model(model)

            precache = cls._precache[model] = {
                'exact': {},
                'words': {},
            }

            for exact, text, object_id in cls.objects.filter(content_type=ct).values_list('exact', 'text', 'object_id'):
                precache['exact' if exact else 'words'].setdefault(text, []).append(object_id)

    @classmethod
    def split(cls, normalized):
        return [
            word for word in cls.SPLIT_RE.split(normalized)
            if word and (word not in cls.STOPWORDS)
        ]

    @classmethod
    def find(cls, match, model, text):
        normalized = normalize(text)

        words = cls.split(normalized)

        if match == 'exact':
            res = cls._find_exact(model, normalized, words)
        elif match == 'words':
            res = cls._find_words(model, words)
        else:
            raise TypeError("Unknown match type %r" % match)

        return SearchResult(model, res)

    @classmethod
    def _find_exact(cls, model, normalized, words):
        search_keys = [normalized]

        if len(words) == 1 and words[0] != normalized:
            search_keys.append(words[0])

        if cls._precache:
            return (
                object_id
                for key in search_keys
                for object_id in cls._precache[model]['exact'].get(key, [])
            )

        ct = ContentType.objects.get_for_model(model)

        return (
            o.object_id
            for o in cls.objects.filter(text__in=search_keys, exact=True, content_type=ct)
        )

    @classmethod
    def _find_words(cls, model, words):
        key = " ".join(words)

        if cls._precache:
            return cls._precache[model]['words'].get(key, [])

        ct = ContentType.objects.get_for_model(model)

        return (
            o.object_id
            for o in cls.objects.filter(text=key, content_type=ct)
        )

    @classmethod
    def get_records(cls):
        for model, fields in cls.SEARCH_FIELDS.items():
            ct = ContentType.objects.get_for_model(model)

            for s in PointSynonym.objects.filter(content_type=ct):
                yield NameSearchIndex(text=normalize(s.title), content_type_id=ct.id, object_id=s.object_id, exact=True)

            for point in model._default_manager.all():
                object_id = point.pk

                for lang in L_field.LANGUAGES:
                    for field, cases in fields.iteritems():
                        for case in cases:

                            text = getattr(point, field)(case=case, lang=lang)

                            if not text:
                                continue

                            normalized = normalize(text)

                            yield NameSearchIndex(text=normalized, content_type_id=ct.id, object_id=object_id, exact=True)

                            words = cls.split(normalized)

                            if not words:
                                continue

                            if len(words) > 1:
                                # Строим индекс с начала фразы, для поиска по начальным словам -
                                for i in xrange(len(words)):
                                    for j in xrange(i, len(words)):
                                        phrase = " ".join(words[i:j + 1])
                                        if not phrase == normalized:
                                            yield NameSearchIndex(text=phrase, content_type_id=ct.id, object_id=object_id, exact=False)

                            elif words[0] != normalized:
                                yield NameSearchIndex(text=words[0], content_type_id=ct.id, object_id=object_id, exact=True)


class DefaultPoint(models.Model):
    """Уточнения в поиске, выбираемые по умолчанию [RASP-4501]"""

    title = TrimmedCharField(verbose_name=_("Название"), max_length=255, unique=True,
                             help_text=_("в нижнем регистре, 'москва'"))
    settlement = models.ForeignKey(Settlement, blank=True, null=True,
                                   verbose_name=_("Город"))
    station = models.ForeignKey(Station, blank=True, null=True,
                                verbose_name=_("Станция"))

    objects = CachingManager(iexact_keys=['title'], use_get_fallback=False)

    def __unicode__(self):
        return self.title

    class Meta:
        verbose_name = _("Уточнение по умолчанию")
        verbose_name_plural = _("Уточнения по умолчанию")
        app_label = 'www'

    @classmethod
    def is_default_title_point(cls, title, point):
        """Проверяет, является ли передаваемая точка дефолтной для названия"""
        try:
            dp = cls.objects.get(title__iexact=title)
        except cls.DoesNotExist:
            pass
        else:
            if isinstance(point, Settlement):
                return dp.settlement_id == point.id

            if isinstance(point, Station):
                return dp.station_id == point.id

        return False
