# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import copy

from travel.avia.library.python.common.models.geo import (
    ExternalDirection, ExternalDirectionMarker,
    Settlement, Station, SuburbanZone,
)


class ExternalDirectionCache(object):
    _markers_by_station_id = None
    _by_station_id = None
    _by_settlement_id = None

    @classmethod
    def precache(cls):
        cls._markers = {}
        cls._by_station_id = {}
        cls._by_settlement_id = {}
        cls._markers_by_station_id = {}

        for o in ExternalDirectionMarker.objects.select_related('station', 'external_direction').all():
            cls._markers_by_station_id.setdefault(o.station_id, []).append(o)

            cls._by_station_id.setdefault(o.station_id, []).append(o.external_direction)

            if o.station.settlement_id:
                cls._by_settlement_id.setdefault(o.station.settlement_id, []).append(o.external_direction)

    @classmethod
    def markers_by_stations(cls, stations):
        if cls._markers_by_station_id is not None:
            result = set()

            for station in stations:
                for marker in cls._markers_by_station_id.get(station.id, []):
                    m = copy.copy(marker)
                    m.station = station

                    result.add(m)

            return result

        return set(ExternalDirectionMarker.objects.filter(station__in=stations))

    @classmethod
    def markers_by_stations_ids(cls, stations_ids):
        if cls._markers_by_station_id is not None:
            r = set()

            for station_id in stations_ids:
                station_markers = cls._markers_by_station_id.get(station_id)

                if station_markers:
                    r.update(station_markers)

            return r

        return set(ExternalDirectionMarker.objects.filter(station__id__in=stations_ids))

    @classmethod
    def by_station(cls, station):
        cache = cls._by_station_id

        if cache is not None:
            return cache.get(station.id, [])

        return ExternalDirection.objects.filter(externaldirectionmarker__station=station)

    @classmethod
    def by_settlement(cls, settlement):
        cache = cls._by_settlement_id

        if cache is not None:
            return cache.get(settlement.id, [])

        # Тут в оригинале был externaldirectionmarker__station__majority_id=1, но этого ограничения
        # не было в прекэше. Поэтому используем единообразный подход по более простому варианту.
        return ExternalDirection.objects.filter(
            externaldirectionmarker__station__settlement=settlement
        )

    @classmethod
    def ids_by_stations_in_bulk(cls, stations):
        cache = cls._by_station_id

        if cache is not None:
            return [(station, set(d.id for d in cache[station.id]))
                    for station in stations
                    if station.id in cache]

        by_station_id = {}

        pairs = ExternalDirectionMarker.objects.filter(station__in=stations) \
            .values_list('station__id', 'external_direction__id')

        for station_id, externaldirection_id in pairs:
            by_station_id.setdefault(station_id, set()).add(externaldirection_id)

        return [(station, by_station_id[station.id])
                for station in stations
                if station.id in by_station_id]

    @classmethod
    def ids_by_settlements_in_bulk(cls, settlements):
        cache = cls._by_settlement_id

        if cache is not None:
            return [(settlement, set(d.id for d in cache[settlement.id]))
                    for settlement in settlements
                    if settlement.id in cache]

        by_settlements_id = {}

        pairs = ExternalDirectionMarker.objects.filter(station__settlement__in=settlements) \
            .values_list('station__settlement__id', 'external_direction__id')

        for settlement_id, externaldirection_id in pairs:
            by_settlements_id.setdefault(settlement_id, set()).add(externaldirection_id)

        return [(s, by_settlements_id[s.id])
                for s in settlements
                if s.id in by_settlements_id]

    @classmethod
    def ids_by_points_in_bulk(cls, points):
        stations = [point for point in points if isinstance(point, Station)]
        settlements = [point for point in points if isinstance(point, Settlement)]

        return cls.ids_by_stations_in_bulk(stations) \
            + cls.ids_by_settlements_in_bulk(settlements)


class SuburbanZoneCache(object):
    _by_settlement_id = None

    @classmethod
    def precache(cls):
        cls._by_settlement_id = {}

        for o in SuburbanZone.objects.all():
            cls._by_settlement_id[o.settlement_id] = o

    @classmethod
    def by_settlement_id(cls, settlement_id):
        cache = cls._by_settlement_id

        if cache is not None:
            return cache.get(settlement_id)

        try:
            return SuburbanZone.objects.get(settlement=settlement_id)
        except SuburbanZone.DoesNotExist:
            return None


class StationCache(object):
    _by_settlement_id = None

    @classmethod
    def precache(cls):
        cls._by_settlement_id = {}

        for o in Station.objects.all_cached():
            if not o.hidden:
                cls._by_settlement_id.setdefault(o.settlement_id, []).append(o)

    @classmethod
    def by_settlement(cls, settlement):
        cache = cls._by_settlement_id

        if cache is not None:
            return cache[settlement.id] if settlement.id in cache else []

        return settlement.station_set.filter(hidden=False)


def have_common_direction(station1_id, station2_id):
    """Определяет, есть ли общее внешнее направление у двух станций"""

    markers_from = ExternalDirectionCache.markers_by_stations_ids([station1_id])
    markers_to = ExternalDirectionCache.markers_by_stations([station2_id])
    directions_from = set(m.external_direction_id for m in markers_from)
    directions_to = set(m.external_direction_id for m in markers_to)

    return not directions_from.isdisjoint(directions_to)
