# -*- coding: utf-8 -*-

from collections import Counter
from itertools import groupby

from django.conf import settings

from travel.avia.library.python.common.models.geo import Settlement, Station
from travel.avia.library.python.common.utils.locations import composeurl
from travel.avia.library.python.common.xgettext.i18n import gettext

from travel.avia.library.python.geosearch.views.pointlist import rearrange_variants


def point_url(point):
    """ Ссылка на город или станцию """
    return composeurl(
        'city' if isinstance(point, Settlement) else 'station',
        args=[point.id]
    )


def group_points(points, client_city):
    headers = {
        1: client_city.L_title(),
        2: client_city.region and client_city.region.L_title() or u'',
        3: client_city.country and client_city.country.L_title() or u'',
        4: gettext(u'Другие страны')
    }

    def grouper(point):
        if hasattr(point, 'settlement') and point.settlement == client_city:
            return 1
        elif hasattr(point, 'region') and client_city.region and point.region == client_city.region:
            return 2
        elif hasattr(point, 'country') and client_city.country and point.country == client_city.country:
            return 3
        else:
            return 4

    return [
        (headers[group_key], list(group_points))
        for group_key, group_points in groupby(points, grouper)
    ]


def get_station_variants(points, client_city, implicit=False):
    title = points.term
    variants = points.variants

    if implicit:
        exact_cities = [
            point
            for point in variants
            if isinstance(point, Settlement) and (
                point.title.lower().replace(u'ё', u'е')
                == title.lower().replace(u'ё', u'е')
            )
        ]

        if len(exact_cities) == 1:
            return [('', [exact_cities[0]])]

    title_count = Counter(point.title for point in variants)

    for point in variants:
        point.url = point_url(point)

        if title_count[point.title] > 1:
            point.has_omonims = True

    return group_points(
        rearrange_variants(variants, client_city, title), client_city)


def get_cities(country, amount=settings.CITIES_IN_COUNTRY_SEARCH):
    """ Формирование списка самых крупных городов по загруженности аэропортов """

    cities = list(Settlement.hidden_manager.filter(country=country).order_by('-threads_amount')[:amount])

    cities = filter(lambda c: not c.disputed_territory, cities)

    stations = Station.objects.filter(settlement__in=cities, t_type=2, hidden=False).order_by('title')

    # Если не нашли аэропортов, то ищем станции с любым типом транспорта
    if not stations:
        stations = Station.objects.filter(settlement__in=cities, hidden=False).order_by('title')

    results = {}

    for s in stations:
        results.setdefault(s.settlement.id, []).append(s)

    return [(city, results[city.id]) for city in cities if city.id in results]
