# -*- coding: utf-8 -*-
import logging
from typing import Any, Optional, List  # noqa

import ujson
from requests import Session, HTTPError
from requests.packages.urllib3.util.retry import Retry
from requests.adapters import HTTPAdapter

from http_geobase.models import GeobaseRegion, GeobaseLinguistics

API_URl = 'http://geobase.qloud.yandex.ru/v1'


class Client(object):
    POSSIBLE_LANGUAGES = ('ru', 'en', 'kk', 'uk', 'tr')

    def __init__(self, api_url=API_URl, user_agent=None, logger=None, http_request_timeout=0.1):
        # type: (str, str, logging.Logger, float) -> None
        self._api_url = api_url
        self._user_agent = user_agent
        self._logger = logger or logging.getLogger(__name__)

        self._session = Session()
        self._session.mount('http://', HTTPAdapter(max_retries=Retry(
            total=3,
            read=3,
            connect=3,
            backoff_factor=0.1,
            status_forcelist=(500, 502, 503, 504),
        )))
        self._http_request_timeout = http_request_timeout
        self._cache = {}

    def region_by_id(self, region_id):
        # type: (int) -> GeobaseRegion
        return GeobaseRegion(ujson.loads(
            self._get('/region_by_id', {'id': region_id}),
        ))

    def region_by_ip(self, ip):
        # type: (str) -> GeobaseRegion
        return GeobaseRegion(ujson.loads(
            self._get('/region_by_ip', {'ip': ip}),
        ))

    def regions(self, ip):
        # type: (str) -> List[GeobaseRegion]
        return map(self.region_by_id, self.parents(self.region_by_ip(ip).id))

    def find_country_id(self, geo_id, crimea_status='ru'):
        # type: (int, str) -> int
        return int(self._get('/find_country', {
            'id': geo_id,
            'crimea_status': crimea_status.lower(),
        }))

    def linguistics(self, geo_id, lang='ru'):
        # type: (int, str) -> GeobaseLinguistics
        if lang not in self.POSSIBLE_LANGUAGES:
            return GeobaseLinguistics({})

        return GeobaseLinguistics(ujson.loads(
            self._get('/linguistics_for_region', {
                'id': geo_id,
                'lang': lang,
            })
        ))

    def clear_cache(self):
        self._cache.clear()

    def parents(self, geo_id, crimea_status='ru'):
        # type (int, str) -> List[int]
        return ujson.loads(self._get('/parents', {
            'id': geo_id,
            'crimea_status': crimea_status.lower(),
        }))

    def regions_by_type(self, region_type):
        # type: (int) -> List[GeobaseRegion]
        return [GeobaseRegion(r) for r in ujson.loads(self._get('/regions_by_type', {'type': region_type}))]

    def _get(self, path, params=None):
        # type: (str, Optional[dict]) -> Any
        url = self._api_url + path

        key = str(url) + str(params)
        if key not in self._cache:
            self._logger.debug('Geobase request "%s" with %s', url, params)

            response = self._session.get(
                url=url,
                params=params,
                headers={
                    'Authorization': self._user_agent,
                },
                timeout=self._http_request_timeout,
            )

            try:
                response.raise_for_status()
            except HTTPError as e:
                self._logger.error('Geobase request error: %s', e)
                raise e

            self._cache[key] = response.content

        return self._cache[key]
