# -*- encoding: utf-8 -*-
import typing
import ujson
from simplejson.decoder import JSONDecodeError

import logging
import requests
from retrying import retry


class IATACorrector:
    LIMIT = 1000

    def __init__(
        self,
        shared_flights_api_base_url='http://shared-flights.production.avia.yandex.net/api/v1',
        logger=None,
        flight_normalizer=None,
    ):
        """
        Iata corrector maps flight numbers to carrier ids

        :param str shared_flights_api_base_url: Base url of iata correction service (shared flights api)
        :param logging.Logger | None logger: override default logger for IATA corrector
        :param ((str)->str) | None flight_normalizer:
        """
        self._shared_flights_base_url = shared_flights_api_base_url
        self._flight_normalizer = flight_normalizer or (lambda flight_number: flight_number)
        self._logger = logger or logging.getLogger(__name__)

    def with_normalizer(self, normalizer):
        """
        Create a copy of this IATA corrector with new normalizer

        :param typing.Callable[[str], str] normalizer:
        :return:
        """
        return self.__class__(
            shared_flights_api_base_url=self._shared_flights_base_url,
            logger=self._logger,
            flight_normalizer=normalizer,
        )

    def flight_number_to_carrier(self, flight_number):
        """
        Converts flight number to map of flight numbers to carriers

        :param typing.Any[str, tuple] flight_number: str or tuples. ex: 'DP 404', ('SU', '234')
        :return: company id
        :rtype: int
        """
        return self.flight_numbers_to_carriers([flight_number])[self._flight_number_to_string(flight_number)]

    def flight_numbers_to_carriers(self, flight_numbers):
        """
        Converts list of flight numbers to map of flight numbers to carriers

        :param typing.Iterable[typing.Any[str, tuple]] flight_numbers: list of str or tuples.
                ex: ['DP 404', ('SU', '234'), ('DP','0404')]
        :return: mapping from flight number to carriers
        :rtype: typing.Dict[str, int]
        """
        result = {}

        company_codes_with_flight_numbers = set([])
        for f in flight_numbers:
            company_codes_with_flight_numbers.add(self._flight_number_to_tuple(f))
            if len(company_codes_with_flight_numbers) == self.LIMIT:
                result.update(self._flight_numbers_to_carriers(company_codes_with_flight_numbers))
                company_codes_with_flight_numbers.clear()

        if len(company_codes_with_flight_numbers) > 0:
            result.update(self._flight_numbers_to_carriers(company_codes_with_flight_numbers))
            company_codes_with_flight_numbers.clear()

        return result

    def _flight_numbers_to_carriers(self, company_codes_with_flight_numbers):
        """
        Converts list of flight numbers to map of flight numbers to carriers

        :param typing.Iterable[typing.Tuple[str]] company_codes_with_flight_numbers: list of tuples.
                ex: [('SU', '234'), ('DP','0404')]
        :return: mapping from flight number to carriers
        :rtype: typing.Dict[str, int]
        """
        flight_numbers = self.get_normalized_flight_numbers(company_codes_with_flight_numbers)

        def retry_on_connection_exc(exc):
            self._logger.warning('Caught %r on requesting Backend', exc)
            return (
                isinstance(exc, requests.ConnectionError) or
                isinstance(exc, ValueError) or
                isinstance(exc, JSONDecodeError)
            )

        @retry(
            retry_on_exception=retry_on_connection_exc,
            stop_max_attempt_number=7,
            wait_exponential_multiplier=200, wait_exponential_max=2000
        )
        def request_backend(base_url, numbers):
            self._logger.info(
                'Try to get company ids for %d flight numbers',
                len(numbers),
            )
            return requests.post(
                url='{}{}'.format(base_url, '/flight-numbers-to-carriers'),
                headers={
                    'Content-type': 'application/json',
                },
                data=ujson.dumps({
                    'flight_numbers': numbers,
                }),
            ).json()

        return request_backend(self._shared_flights_base_url, flight_numbers)

    def get_normalized_flight_numbers(self, company_codes_with_flight_numbers):
        """
        Normalize list of tuples with flight comapny code and flight number. ex: [('SU', '1256')]
        Normazlied using function provided in constructor

        :param company_codes_with_flight_numbers:
        :return: list of normalized flight numbers
        :rtype: typing.List[str]
        """
        flight_numbers = []
        for code_and_number in company_codes_with_flight_numbers:
            normalized = self._normalize(code_and_number)
            if normalized:
                flight_numbers.append(normalized)

        return list(set(flight_numbers))

    def _normalize(self, code_and_number):
        """
        Nomralizes single carrier code and flight number pair using provided normalizer function

        :param str code_and_number: tuple with company code and flight number. ex: ('SU', '1256')
        :return: normalized flight number
        :rtype: str | None
        """
        if not code_and_number:
            return None
        try:
            company_code, flight_number = code_and_number
            return company_code + ' ' + self._flight_normalizer(flight_number)
        except ValueError:
            self._logger.exception('Unexpected company code and number: %s', code_and_number)
        except AttributeError:
            self._logger.exception('Can\'t normalize flight number %s', code_and_number)

        return None

    @staticmethod
    def _flight_number_to_tuple(flight_number):
        """
        :param typing.Any[str, typing.Iterable[str]] flight_number:
        :rtype: typing.Iterable[str]
        """
        if hasattr(flight_number, 'split'):
            flight_number = flight_number.split(' ') if ' ' in flight_number else ('', flight_number)

        return tuple(flight_number)

    @staticmethod
    def _flight_number_to_string(flight_number):
        """
        :param typing.Any[str, typing.Iterable[str]] flight_number:
        :rtype: str
        """
        if hasattr(flight_number, 'split'):
            return ' ' + flight_number if ' ' not in flight_number else flight_number
        return ' '.join(flight_number)
