# coding=utf-8
from __future__ import unicode_literals

import logging
import random
from collections import namedtuple

from typing import Optional, Union

logger = logging.getLogger(__name__)


class YtCache(object):
    class_cache = dict()

    def __init__(self, yt, yt_table):
        self.yt = yt
        self.yt_table = yt_table
        self.cache = None

    def precache(self, key_column, value_putter):
        if self.cache is None:
            cache = self.class_cache_lookup(key_column)
            if cache is not None:
                self.cache = cache
                return
            cache = {}
            row_iterator = self.yt.read_table(self.yt_table)
            for s in row_iterator:
                key = s[key_column]
                value_putter(cache, key, s)
            self.class_cache_put(key_column, cache)
            self.cache = cache

    def _class_cache_key(self, key_column):
        return self.yt_table, key_column

    def class_cache_lookup(self, key_column):
        return YtCache.class_cache.get(self._class_cache_key(key_column))

    def class_cache_put(self, key_column, value):
        YtCache.class_cache[self._class_cache_key(key_column)] = value


class Station(YtCache):
    def __init__(self, yt, station_table='//home/rasp/reference/station'):
        super(Station, self).__init__(yt, station_table)

    @staticmethod
    def item_putter(dic, k, v):
        dic[int(k[1:])] = v  # strip s from station id ("s1235")

    @property
    def by_id(self):
        self.precache('id', self.item_putter)
        return self.cache


class Settlement(YtCache):
    def __init__(self, yt, station_table='//home/rasp/reference/settlement'):
        super(Settlement, self).__init__(yt, station_table)

    @staticmethod
    def item_putter(dic, k, v):
        dic[int(k[1:])] = v  # strip c from settlement id ("c1235")

    @property
    def by_id(self):
        self.precache('id', self.item_putter)
        return self.cache


class Company(YtCache):
    def __init__(self, yt, company_table='//home/rasp/reference/company'):
        super(Company, self).__init__(yt, company_table)

    @property
    def by_id(self):
        self.precache('id', put_single_item)
        return self.cache


class Partner(YtCache):
    def __init__(self, yt, company_table='//home/rasp/reference/partner'):
        super(Partner, self).__init__(yt, company_table)

    @property
    def by_id(self):
        self.precache('partner_id', put_single_item)
        return self.cache


class BadAvatarUrl(ValueError):
    pass


Dimensions = namedtuple('Dimensions', 'width height')


class SettlementBigImage(YtCache):
    ANY = 'ANY'

    def __init__(self, yt, company_table='//home/rasp/reference/adminsettlementbigimage'):
        super(SettlementBigImage, self).__init__(yt, company_table)

    @staticmethod
    def item_putter(dic, k, v):
        if k and len(k) > 1:
            dic[int(k[1:])] = v  # strip c from settlement id ("c1235")
        else:
            k = None
            s = dic.get(k, set())
            s.add(v['url2'])
            dic[k] = s

    @property
    def by_settlement_id(self):
        self.precache('settlement', self.item_putter)
        return self.cache

    def _orig_to_square_or_empty(self, orig):
        # type: (Optional[str])->str
        if not orig:
            return ''
        if not orig.endswith('/orig'):
            raise BadAvatarUrl('Avatar url should end with /orig', orig)
        return orig[:-len('orig')] + 'travel-marketing-square'

    def get_default_original(self):
        # type: ()->str
        return random.choice(list(self.by_settlement_id.get(None, [])))

    def get_default(self, defaults_filter=None):
        # type: (Optional[Union[set, str]]) -> str
        """
        Choose default picture with ability to filter
        :param defaults_filter: If None - return empty string, if set - filter images without settlements using provided set,
        if 'ANY' - use all images without settlements
        :return:
        """
        if defaults_filter is None:
            return ''
        settlement_free = set(self.by_settlement_id.get(None, set()))
        if isinstance(defaults_filter, set):
            settlement_free = settlement_free & defaults_filter
            if len(settlement_free) != len(defaults_filter):
                logger.warning('Some defaults were removed from admin. Consider revising')
        elif defaults_filter == SettlementBigImage.ANY:
            pass  # do not filter, but leave for code clarity
        if not settlement_free:
            return ''
        return self._orig_to_square_or_empty(random.choice(list(settlement_free)))

    def get_marketing_square(self, settlement_id, defaults_filter=None):
        # type: (int, Optional[Union[set,str]])->str
        """
        :param settlement_id: ex.: 213
        :param defaults_filter: if defaults_policy is None (default) - the default value will be an empty string;
        if default_policy is a string 'ANY' - the default value will be randomly chosen from all images that are not linked
        to any settlement;
        if default_poicy is a set of strings - this set acts as a filter, return value will be randomly chosend from
        (all images set & default_policy set)
        :return:
        """
        if settlement_id not in self.by_settlement_id:
            return self.get_default(defaults_filter)
        image_dict = self.by_settlement_id[settlement_id]
        orig = image_dict.get('url2')
        return self._orig_to_square_or_empty(orig)


def put_single_item(dic, key, value):
    dic[key] = value
