# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import datetime  # noqa
from itertools import izip
from collections import namedtuple

import pytz

from travel.avia.library.python.references.station  import StationCache  # noqa

TripInfo = namedtuple('TripInfo', ['origin', 'destination', 'date_forward', 'date_backward', 'trip_type'])


class AviaFlightRouteHelper(object):
    def __init__(self, station_cache):
        self.station_cache = station_cache  # type: StationCache

    def localize_datetime(self, flights):
        for flight in flights:
            flight['departure_dt'] = self._localize_datetime(
                flight['departure_dt'],
                self._timezone(flight['from'])
            )
            flight['arrival_dt'] = self._localize_datetime(
                flight['arrival_dt'],
                self._timezone(flight['to'])
            )

    @staticmethod
    def _localize_datetime(naive_dt, tz_name):
        # type: (datetime.datetime, basestring) -> datetime.datetime
        return pytz.timezone(tz_name).localize(naive_dt)

    def fillin_airports(self, order):
        """
        Принимаем словарик заказа и заполняем начальную и конечную точку маршрута.
        Информацию о структуре входных данных смотрим в _get_airports

        :param dict[basestring, Any] order: заказ с информацией о сегментах перелета.
        """
        trip_info = self._get_trip_info(order)
        order['origin'] = trip_info.origin
        order['destination'] = trip_info.destination
        order['trip_type'] = trip_info.trip_type
        return order

    def fillin_trip_info(self, order):
        """
        Принимаем словарик заказа и заполняем начальную и конечную точку маршрута и даты туда/обратно
        Информацию о структуре входных данных смотрим в _get_airports

        :param dict[basestring, Any] order: заказ с информацией о сегментах перелета.
        """
        trip_info = self._get_trip_info(order)
        order['origin'] = trip_info.origin
        order['destination'] = trip_info.destination
        order['trip_type'] = trip_info.trip_type
        order['date_forward'] = trip_info.date_forward
        order['date_backward'] = trip_info.date_backward
        return order

    def _get_trip_info(self, order):
        """
        Получаем из неупорядоченного списка сегментов пункт начала маршрута и пункт конца.
        Проверяем, что это маршрут закольцованный и если так, то ищем середину
        Проверяем, что это один цельный маршрут.

        Формат структуры по ключу flights:
        [{
            from: ..., -- IATA код аэропорта отлета

            to: ...,  -- IATA код аэропорта прилета

            departure_dt: ...,  --дата и время отлета datetime

            arrival_dt: ...,  --дата и время прилета datetime
        }]

        :param dict[basestring, list[dict[basestring, Any]]] order: Заказ, в котором по ключу flights лежит список полетов
        :rtype: TripInfo
        """
        flights = sorted(order['flights'], key=lambda flight: flight['departure_dt'])

        if not flights:
            raise Exception('No segments found')

        orig = flights[0]['from']
        dest = flights[-1]['to']

        date_forward = flights[0]['departure_dt'].strftime('%Y-%m-%d').decode('utf8')
        date_backward = ''
        if self._is_round_trip(flights):
            dest = self._most_delayed_airport(flights)
            backward_segment = next((f for f in flights if f['from'] == dest), None)
            if backward_segment:
                date_backward = backward_segment['departure_dt'].strftime('%Y-%m-%d').decode('utf8')
            trip_type = 'roundtrip'
        else:
            trip_type = 'oneway'

        if not self._is_continuous(flights):
            trip_type = 'openjaw'

        return TripInfo(
            origin=orig,
            destination=dest,
            trip_type=trip_type,
            date_forward=date_forward,
            date_backward=date_backward,
        )

    def _is_round_trip(self, flights):
        """
        Определяем, прилетели ли мы в тот же город, из которого улетали

        :param list[dict[basestring,Any]] flights: Список словарей с ключами from и to и значениями в виде IATA кодов
        :rtype: bool
        :return: True если вернулись откуда прилетели, иначе False
        """
        if not flights or len(flights) == 1:
            return False
        orig = flights[0]['from']
        dest = flights[-1]['to']
        if orig is None or dest is None:
            return False
        if orig == dest:
            return True
        city_orig = self._city(orig)
        city_dest = self._city(dest)
        if city_orig is None or city_dest is None:
            return False
        if city_orig == city_dest:
            return True
        return False

    def _is_continuous(self, flights):
        """
        Определяем, является ли маршрут географически непрерывным,
        то есть нет прилета в один город, а вылета из другого

        :param  list[dict[basestring, Any]] flights: список словарей полетов по маршруту с ключами from и to.
        Значения ключей from и to - IATA/SIRENA коды аэропорта
        :return: True если полет нерперывный, иначе False
        """
        if not flights:
            raise Exception('No flights found')
        if len(flights) == 1:
            return True
        for (past, future) in izip(flights, flights[1:]):
            inport = past['to']
            outport = future['from']
            if inport is None or outport is None:
                return False

            city_inport = self._city(inport)
            city_outport = self._city(outport)

            if city_inport is None or city_outport is None:
                return False

            if inport != outport and city_inport != city_outport:
                return False
        return True

    @staticmethod
    def _most_delayed_airport(flights):
        """
        Находим среди всех перелетов аэропорт, в котором находились дольше всего
        Необходимо при поиске направления в случае round trip

        :param list[dict[basestring, Any]] flights: список словарей с ключами from (IATA/SIRENA), to (IATA/SIRENA), deprature_dt (datetime),
        arrival_dt (datetime)
        :rtype: basestring
        :return: Аэропорт, в котором была наибольшая задержка между перелетами
        """
        if not flights:
            raise Exception('No flights found')
        if len(flights) == 1:
            raise Exception('Cannot determine for a single flight')

        max_delay = None
        max_delay_airport = None
        for (past, future) in izip(flights, flights[1:]):
            delay = future['departure_dt'] - past['arrival_dt']
            if max_delay is None or delay > max_delay:
                max_delay = delay
                if future['from'] is not None:
                    max_delay_airport = future['from']
                else:
                    max_delay_airport = past['to']
        return max_delay_airport

    def _city(self, airport_code):
        return self.station_cache.settlement_id_by_code(airport_code, raise_on_unknown=False)

    def _timezone(self, airport_code):
        try:
            return self.station_cache.station_timezone_by_station_code(airport_code)
        except KeyError:
            return "UTC"
