# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import socket
import uuid
from collections import namedtuple

from infra.yp_service_discovery.api import api_pb2
from infra.yp_service_discovery.python.resolver.resolver import Resolver

from travel.avia.library.python.price_prediction import STAGE_BY_ENVIRONMENT

logger = logging.getLogger(__name__)


class Endpoint(namedtuple('Endpoint', ('host', 'port'))):
    @property
    def target(self):
        return '{host}:{port}'.format(host=self.host, port=self.port)


def discover(environment, timeout=5.0):
    client_name = 'avia:{}'.format(socket.gethostname())
    resolver = Resolver(client_name=client_name, timeout=timeout)
    endpoints = []
    for dc in ('sas', 'man', 'vla', 'iva', 'myt'):
        request = api_pb2.TReqResolveEndpoints(
            cluster_name=dc,
            endpoint_set_id=STAGE_BY_ENVIRONMENT[environment],
            client_name=client_name,
            ruid='resolve-avia-pp-in-{}-{}'.format(dc, uuid.uuid4().hex),
            label_selectors=[],
        )
        try:
            result = resolver.resolve_endpoints(request)
        except Exception:
            logger.exception('error discover price-prediction in %s', dc)
        else:
            if result.endpoint_set and result.endpoint_set.endpoints:
                endpoints.extend(result.endpoint_set.endpoints)

    endpoints = [e for e in endpoints if e.ready] or endpoints or []
    result = [Endpoint(endpoint.fqdn, endpoint.port) for endpoint in endpoints]
    logger.info('discover endpoints %s', result)
    return result
