from travel.library.python.yp.endpoints import YpEndpoints

from travel.avia.library.python.proxy_pool.proxy_pool import ProxyPool


class DeployProxyPool(ProxyPool):
    CLIENT_NAME = 'avia'
    DEPLOY_ENDPOINT_SET_ID = 'avia-partners-proxy-{env}.proxy'
    DEVELOPMENT_ENDPOINT_SET_ID = 'avia-partners-proxy-unstable.proxy'
    DATACENTERS = {
        'sas': ['sas', 'vla', 'vlx', 'myt', 'iva', 'man'],
        'vla': ['vla', 'vlx', 'sas', 'iva', 'myt', 'man'],
        'iva': ['iva', 'myt', 'sas', 'vla', 'vlx', 'man'],
        'myt': ['myt', 'iva', 'sas', 'vla', 'vlx', 'man'],
        'man': ['man', 'vla', 'vlx', 'sas', 'iva', 'myt'],
        'vlx': ['vlx', 'vla', 'sas', 'iva', 'myt', 'man'],
    }

    def __init__(self, environment, datacenter=None, login=None, password=None, logger=None):
        super(DeployProxyPool, self).__init__(login, password, logger)
        if environment in ('development', 'dev'):
            self._endpoint_set_id = self.DEVELOPMENT_ENDPOINT_SET_ID
        else:
            self._endpoint_set_id = self.DEPLOY_ENDPOINT_SET_ID.format(env=environment)
        self._datacenter = datacenter.lower() if datacenter else None

        if self._datacenter not in self.DATACENTERS:
            raise RuntimeError('Unknown datacenter "{}", available datacenters: {}'.format(
                self._datacenter,
                ', '.join(self.DATACENTERS.keys()),
            ))

    def _resolve_proxies(self):
        endpoints = self._resolve_endpoints(self._endpoint_set_id)
        for dc in self.DATACENTERS[self._datacenter]:
            if dc in endpoints:
                self._logger.info(
                    'From "%s" to "%s" use %d proxies: %s', self._datacenter, dc, len(endpoints[dc]), endpoints[dc]
                )
                return endpoints[dc]

        raise RuntimeError('No proxy available for client={} and endpoint={}'.format(
            self.CLIENT_NAME,
            self._endpoint_set_id,
        ))

    def _resolve_endpoints(self, endpoint_set_id):
        self._logger.info('Trying to resolve deploy endpoint: %s', endpoint_set_id)
        all_endpoints = YpEndpoints(
            self.CLIENT_NAME,
            endpoint_set_id,
            datacenters=self.DATACENTERS.keys(),
        ).get_ip6_addresses_by_dc()

        result = {
            dc: endpoints
            for dc, endpoints in all_endpoints.items() if len(endpoints) > 0
        }
        self._logger.info('Resolved deploy endpoints for datacenters: %s', ', '.join(result.keys()))

        return result
