import logging
from itertools import cycle


class ProxyHost(object):
    HTTP_PORT = 80
    FTP_PORT = 8021

    def __init__(self, host, login=None, password=None):
        self._host = host
        self._login = login
        self._password = password

    def get_http_uri(self):
        # type: () -> str
        proxy_auth = self._get_proxy_auth_str()
        return 'http://{proxy_auth}[{host}]:{port}'.format(
            proxy_auth='{}@'.format(proxy_auth) if proxy_auth else '',
            host=self._host,
            port=self.HTTP_PORT,
        )

    def get_requests_proxies(self):
        # type: () -> dict
        uri = self.get_http_uri()
        return {
            'all': uri,
        }

    def get_ftp_host(self):
        # type: () -> str
        return self._host

    def get_ftp_user(self, ftp_login, ftp_host):
        # type: (str, str) -> str
        proxy_auth = self._get_proxy_auth_str()
        return '{proxy_auth}{ftp_login}@{ftp_host}'.format(
            proxy_auth='{}:'.format(proxy_auth) if proxy_auth else '',
            ftp_login=ftp_login,
            ftp_host=ftp_host,
        )

    def _get_proxy_auth_str(self):
        return '{}{}'.format(
            '{}'.format(self._login) if self._login else '',
            ':{}'.format(self._password) if self._login and self._password else '',
        ) if self._login or self._password else ''

    def __repr__(self):
        return '{}: host={}, login={}'.format(type(self), self._host, self._login)


class ProxyPool(object):
    def __init__(self, login=None, password=None, logger=None, proxies=None):
        self._login = login
        self._password = password
        self._logger = logger or logging.getLogger(__name__)

        if proxies:
            self._set_proxies(proxies)
        else:
            self._proxies = None
            self._proxies_cycle = None

    def _resolve_proxies(self):
        return []

    def _set_proxies(self, proxies):
        self._proxies = [ProxyHost(host=proxy, login=self._login, password=self._password) for proxy in set(proxies)]
        self._proxies_cycle = cycle(self._proxies)

    def reload_proxies(self):
        self._set_proxies(self._resolve_proxies())

    def get_proxy(self):
        # type: () -> ProxyHost
        if self._proxies_cycle is None:
            self._set_proxies(self._resolve_proxies())
        return next(self._proxies_cycle) if self._proxies_cycle else None

    def __repr__(self):
        return '{}: proxies={}, login={}'.format(type(self), self._proxies, self._login)
